package com.bcxin.tenant.domain.repository.customize;
import com.bcxin.Infrastructures.UnitWork;
import com.bcxin.Infrastructures.exceptions.BadTenantException;
import com.bcxin.tenant.domain.entities.DepartmentAdminEntity;
import com.bcxin.tenant.domain.entities.DepartmentEntity;
import com.bcxin.tenant.domain.entities.EmployeeEntity;
import com.bcxin.tenant.domain.repositories.DepartmentAdminRepository;
import org.hibernate.Session;
import org.springframework.stereotype.Repository;
import org.springframework.util.CollectionUtils;

import javax.persistence.EntityManager;
import javax.persistence.Query;
import javax.persistence.TypedQuery;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

@Repository
public class DepartmentAdminRepositoryImpl implements DepartmentAdminRepository {
    private final EntityManager entityManager;
    private final UnitWork unitWork;

    public DepartmentAdminRepositoryImpl(EntityManager entityManager, UnitWork unitWork) {
        this.entityManager = entityManager;
        this.unitWork = unitWork;
    }


    @Override
    public void create(DepartmentAdminEntity departmentAdmin) {
        this.entityManager.unwrap(Session.class).persist(departmentAdmin);
        this.entityManager.flush();
    }

    @Override
    public void update(List<DepartmentAdminEntity> departmentAdminList) {
        for (DepartmentAdminEntity departmentAdminEntity : departmentAdminList) {
            this.entityManager.unwrap(Session.class).update(departmentAdminEntity);
        }
    }

    @Override
    public void batchCreate(Collection<EmployeeEntity> employees, Collection<DepartmentEntity> departments) {
        if (CollectionUtils.isEmpty(employees) || CollectionUtils.isEmpty(departments)) {
            throw new BadTenantException("职员/部门不能为空");
        }

        try {
            this.unitWork.executeTran(() -> {
                for (EmployeeEntity employee : employees) {
                    Query deleteQuery =
                            this.entityManager.createQuery("delete from DepartmentAdminEntity t where t.employee.id=?1 and t.department.id in (?2)");
                    deleteQuery.setParameter(1, employee.getId());
                    deleteQuery.setParameter(2, departments.stream().map(ix -> ix.getId()).collect(Collectors.toList()));
                    deleteQuery.executeUpdate();
                    for (DepartmentEntity department : departments) {
                        DepartmentAdminEntity departmentAdmin = DepartmentAdminEntity.create(department, employee);


                        this.entityManager.unwrap(Session.class).persist(departmentAdmin);
                    }
                }
            });
        } catch (Exception ex) {
            throw new BadTenantException("系统异常, 请联系管理员!");
        }
    }

    @Override
    public void update(EmployeeEntity employee, Collection<DepartmentEntity> departments) {
        if (employee == null || CollectionUtils.isEmpty(departments)) {
            throw new BadTenantException("职员/部门不能为空");
        }

        try {
            this.unitWork.executeTran(() -> {
                Query deleteQuery =
                        this.entityManager.createQuery("delete from DepartmentAdminEntity t where t.employee.id=?1");
                deleteQuery.setParameter(1, employee.getId());
                deleteQuery.executeUpdate();
                for (DepartmentEntity department : departments) {
                    DepartmentAdminEntity departmentAdmin = DepartmentAdminEntity.create(department, employee);
                    this.entityManager.unwrap(Session.class).persist(departmentAdmin);
                }
            });
        } catch (Exception ex) {
            throw new BadTenantException("系统异常, 请联系管理员!");
        }
    }

    @Override
    public void batchDelete(Collection<String> employeeIds) {
        if (CollectionUtils.isEmpty(employeeIds)) {
            throw new BadTenantException("要删除的职员不能为空!");
        }

        this.unitWork.executeTran(() -> {
            Query deleteQuery =
                    this.entityManager.createQuery("delete from DepartmentAdminEntity u where u.employee.id in (?1)");

            deleteQuery.setParameter(1, employeeIds);

            deleteQuery.executeUpdate();
        });
    }

    @Override
    public Collection<DepartmentAdminEntity> getByIds(String organizationId, Collection<String> ids) {
        TypedQuery<DepartmentAdminEntity> departmentAdminEntityTypedQuery
                = this.entityManager.createQuery("select d from DepartmentAdminEntity d where d.organization.id=?1 and d.id in (?2)",
                DepartmentAdminEntity.class);

        departmentAdminEntityTypedQuery.setParameter(1, organizationId);
        departmentAdminEntityTypedQuery.setParameter(2, ids);

        return departmentAdminEntityTypedQuery.getResultList();
    }

    @Override
    public Collection<DepartmentAdminEntity> getByEmployeeIds(String organizationId, Collection<String> employeeIds) {
        TypedQuery<DepartmentAdminEntity> departmentAdminEntityTypedQuery
                = this.entityManager.createQuery("select d from DepartmentAdminEntity d where d.organization.id=?1 and d.employee.id in (?2)",
                DepartmentAdminEntity.class);

        departmentAdminEntityTypedQuery.setParameter(1, organizationId);
        departmentAdminEntityTypedQuery.setParameter(2, employeeIds);

        return departmentAdminEntityTypedQuery.getResultList();
    }

    @Override
    public List<DepartmentAdminEntity> getByDepartmentId(String departmentId) {
        TypedQuery<DepartmentAdminEntity> departmentAdminEntityTypedQuery
                = this.entityManager.createQuery("select d from DepartmentAdminEntity d where d.department.id=?1 ",
                DepartmentAdminEntity.class);

        departmentAdminEntityTypedQuery.setParameter(1, departmentId);

        return departmentAdminEntityTypedQuery.getResultList();
    }
}
