package com.bcxin.rest.web.apis.utils;

import cn.hutool.http.HttpUtil;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import javax.servlet.http.HttpServletRequest;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Map;


@Slf4j
public class IPUtils {

    /**
     * 获取IP地址
     *
     * 使用Nginx等反向代理软件， 则不能通过request.getRemoteAddr()获取IP地址
     * 如果使用了多级反向代理的话，X-Forwarded-For的值并不止一个，而是一串IP地址，X-Forwarded-For中第一个非unknown的有效IP字符串，则为真实IP地址
     */
    public static String getIpAddress(HttpServletRequest request) {
        String ip = null;
        try {
            ip = request.getHeader("x-forwarded-for");
            if (StringUtils.isBlank(ip) || "unknown".equalsIgnoreCase(ip)) {
                ip = request.getHeader("Proxy-Client-IP");
            }
            if (StringUtils.isBlank(ip) || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
                ip = request.getHeader("WL-Proxy-Client-IP");
            }
            if (StringUtils.isBlank(ip) || "unknown".equalsIgnoreCase(ip)) {
                ip = request.getHeader("HTTP_CLIENT_IP");
            }
            if (StringUtils.isBlank(ip) || "unknown".equalsIgnoreCase(ip)) {
                ip = request.getHeader("HTTP_X_FORWARDED_FOR");
            }
            if (StringUtils.isBlank(ip) || "unknown".equalsIgnoreCase(ip)) {
                ip = request.getRemoteAddr();
            }
        } catch (Exception e) {
            log.error("IPUtils ERROR：",e);
        }

        //对于通过多个代理的情况，第一个IP为客户端真实IP,多个IP按照','分割
        if(!StringUtils.isBlank(ip) && ip.length() > 15) {
            if(ip.indexOf(",") > 0) {
                ip = ip.substring(0, ip.indexOf(","));
            }
        }

        return ip;
    }

    public static Map<String,String> ipMap = Maps.newHashMap();

    /**
     * 判断 IP 是否属于陕西省或北京市
     * @param ip IP 地址
     * @return true 表示是陕西或北京，false 表示不是
     */
    public static String isFromShaanxiOrBeijing(String ip) {
        if (StringUtils.isBlank(ip) || "unknown".equalsIgnoreCase(ip)) {
            return "";
        }
        // 校验IP格式（防止无效IP）
        if (!isValidIp(ip)) {
            return "";
        }

        // 对IP进行URL编码，防止特殊字符导致请求失败
        String encodedIp;
        try {
            encodedIp = URLEncoder.encode(ip, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            log.error("IP编码失败：", e);
            return "";
        }
        String region = ipMap.get(ip);

        log.error("本轮之前ip已添加序列总数：", ipMap.size());
        if(StringUtils.isBlank(region)){
            if(ipMap.size()>100){
                log.error("IP编码不在预期序列：", ip);
                return "";
            }
            try {
                String response = HttpUtil.get("https://ipinfo.io/" + encodedIp + "/json");
                if (StringUtils.isBlank(response)) {
                    return "";
                }
                JSONObject jsonObject = JSONUtil.parseObj(response);
                region = jsonObject.getStr("region");
                if (StringUtils.isNotEmpty(region)) {
                    ipMap.put(ip, region);
                }
            } catch (Exception e) {
                log.error("IP归属地查询异常：", e);
                region = "";
            }
        }


        if ("Beijing".equals(region)) {
            return "11";
        } else if ("Shaanxi".equals(region)) {
            return "61";
        } else if ("Hunan".equals(region)) {
            return "43";
        }
        return "";
    }

    /**
     * 简单的IP格式校验（IPv4）
     */
    private static boolean isValidIp(String ip) {
        String ipPattern = "^((25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\\.){3}(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$";
        return ip.matches(ipPattern);
    }
}