package com.bcxin.rest.web.apis.controllers;

import com.bcxin.Infrastructures.Pageable;
import com.bcxin.Infrastructures.components.RetryProvider;
import com.bcxin.api.interfaces.commons.ConfigDictRpcProvider;
import com.bcxin.api.interfaces.commons.responses.CompanyInfoCompleteResponse;
import com.bcxin.api.interfaces.commons.responses.CompanyNameExistResponse;
import com.bcxin.api.interfaces.commons.responses.CompanyPromoterResponse;
import com.bcxin.api.interfaces.commons.responses.CompanyResponse;
import com.bcxin.api.interfaces.tenants.EmployeeRpcProvider;
import com.bcxin.api.interfaces.tenants.OrganizationRpcProvider;
import com.bcxin.api.interfaces.tenants.criterias.OrganizationCriteria;
import com.bcxin.api.interfaces.tenants.requests.organizations.*;
import com.bcxin.api.interfaces.tenants.responses.*;
import com.bcxin.rest.web.apis.utils.FileUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import java.util.List;

/**
 * 先统一再Service进行验证
 */
//@Validated
@RestController
@RequestMapping("/tenant/organizations")
@Api(value = "企业管理api", tags = "企业管理api")
public class OrganizationController extends ControllerAbstract {
    private final OrganizationRpcProvider organizationRpcProvider;
    private final EmployeeRpcProvider employeeRpcProvider;
    private final RetryProvider retryProvider;


    public OrganizationController(OrganizationRpcProvider organizationRpcProvider,
                                  EmployeeRpcProvider employeeRpcProvider, RetryProvider retryProvider) {
        this.organizationRpcProvider = organizationRpcProvider;
        this.employeeRpcProvider = employeeRpcProvider;
        this.retryProvider=retryProvider;
    }
    @ApiOperation(value = "邀请注册", response = CompanyResponse.class)
    @GetMapping("/pre_registration/{number}")
    public ResponseEntity<CompanyResponse> preRegistration(@PathVariable String number) {
        TenantOrganizationResponse response = this.organizationRpcProvider.getOrganizationReqewsts(number);
        CompanyResponse companyResponse = new CompanyResponse();
        if (response.getContent() != null) {
            response.getContent().setCompanyValue(companyResponse);
            companyResponse.setIndustryCode(response.getIndustryCode());
            companyResponse.setInstitutionalCode(response.getInstitutionalCode());
        }
        return this.ok(companyResponse);
    }

    @ApiOperation(value = "判断企业名称是否存在", response = CompanyNameExistResponse.class)
    @PostMapping("/name/exist")
    public ResponseEntity<CompanyNameExistResponse> nameExist(@RequestBody OrganizationCriteria criteria) {
        CompanyNameExistResponse response = organizationRpcProvider.nameExist(criteria.getName());
        return this.ok(response);
    }

    @ApiOperation(value = "注册企业信息", response = String.class)
    @PostMapping("/registration")
    public ResponseEntity post(@Valid @RequestBody CompanyRegistrationRequest request) {
        String comId = this.organizationRpcProvider.register(request);
        return this.ok(comId);
    }

    @ApiOperation(value = "获取企业详情", response = CompanyResponse.class)
    @GetMapping("/{organizationId}")
    public ResponseEntity<CompanyResponse> get(@PathVariable String organizationId) {
        CompanyResponse response = organizationRpcProvider.get(organizationId);

        return this.ok(response);
    }

    @ApiOperation(value = "判断企业是否分销商", response = CompanyResponse.class)
    @GetMapping("/{organizationId}/is/promoter")
    public ResponseEntity<CompanyPromoterResponse> isOrganizationPromoter(@PathVariable String organizationId) {
        CompanyPromoterResponse response = organizationRpcProvider.isOrganizationPromoter(organizationId);
        return this.ok(response);
    }

    @ApiOperation(value = "获取企业详情", response = CompanyResponse.class)
    @GetMapping("/{organizationId}/getDetail")
    public ResponseEntity<CompanyResponse> getDetail(@PathVariable String organizationId,String approveOrganId) {
        CompanyResponse response = organizationRpcProvider.getDetail(approveOrganId);

        return this.ok(response);
    }

    @ApiOperation(value = "审批企业", response = OrganizationResponse.class)
    @PutMapping("/{organizationId}/approved")
    public ResponseEntity approved(@PathVariable String organizationId,
                                   @RequestBody ApproveCompanyRequest request) {
        this.organizationRpcProvider.approve(organizationId, request);

        return ok();
    }


    @ApiOperation(value = "审批修改企业信息", response = String.class)
    @PostMapping("/{organizationId}/approved/update")
    public ResponseEntity approvedUpdate(@PathVariable String organizationId, @RequestBody CompanyApprovedUpdateRequest request) {
        request.setId(organizationId);
        this.organizationRpcProvider.approvedUpdate(request);
        return this.ok();
    }

    @ApiOperation(value = "搜索企业信息", response = OrganizationResponse.class)
    @PostMapping("/search")
    public ResponseEntity search(@RequestBody OrganizationCriteria criteria) {
        Pageable<OrganizationResponse> data =
                this.organizationRpcProvider.search(criteria);


        return this.ok(data);
    }

    @ApiOperation(value = "导出企业信息", response = OrganizationResponse.class)
    @RequestMapping("/{organizationId}/exportOrgExcel")
    public void exportOrgExcel(@RequestBody OrganizationCriteria criteria, HttpServletResponse response) throws Exception {
        String filePath = this.organizationRpcProvider.createOrgExcel(criteria);
        response.setContentType(MediaType.APPLICATION_OCTET_STREAM_VALUE);
        FileUtils.setAttachmentResponseHeader(response, "组织信息.xlsx");
        FileUtils.writeBytes(filePath, response.getOutputStream(),retryProvider);
        FileUtils.deleteFile(filePath);
    }

   
    @Deprecated
    @ApiOperation(value = "精确搜索企业信息", response = OrganizationExactSearchResponse.class)
    @PostMapping("/{organizationId}/exactSearch")
    public ResponseEntity exactSearch_Removed(@RequestBody OrganizationRequest request) {
        return exactSearch(request);
    }
    
    @ApiOperation(value = "精确搜索企业信息", response = OrganizationExactSearchResponse.class)
    @PostMapping("/{organizationId}/exact/search")
    public ResponseEntity exactSearch(@RequestBody OrganizationRequest request) {
        request.validate();
        List<OrganizationExactSearchResponse> data = this.organizationRpcProvider.exactSearch(request);
        return this.ok(data);
    }

    @ApiOperation(value = "修改企业信息-经纬度(组织管理员修改)", response = String.class)
    @PostMapping("/{organizationId}/update")
    public ResponseEntity updateByAdmin(@PathVariable String organizationId,@Valid @RequestBody CompanyUpdateCoordinateRequest request) {
        CompanyUpdateRequest updateRequest = new CompanyUpdateRequest();
        updateRequest.setId(organizationId);
        updateRequest.setLatitude(request.getLatitude());
        updateRequest.setLongitude(request.getLongitude());
        updateRequest.setLocationAddress(request.getAddress());
        updateRequest.setLogoPath(request.getLogoPath());
        this.organizationRpcProvider.update(updateRequest);
        return this.ok();
    }

    @ApiOperation(value = "修改企业信息", response = String.class)
    @PostMapping("/update/{organizationId}")
    public ResponseEntity update(@PathVariable String organizationId, @RequestBody CompanyUpdateRequest request) {
        request.setId(organizationId);
        this.organizationRpcProvider.update(request);
        return this.ok();
    }

    @ApiOperation(value = "修改企业监管机构", response = String.class)
    @PostMapping("/update/supervise/{organizationId}")
    public ResponseEntity updateSupervise(@PathVariable String organizationId, @Valid @RequestBody CompanyUpdateSuperviseRequest request) {
        CompanyUpdateRequest updateRequest = new CompanyUpdateRequest();
        updateRequest.setId(organizationId);
        updateRequest.setPlaceOfSupervise(request.getPlaceOfSupervise());
        updateRequest.setSuperviseDepartId(request.getSuperviseDepartId());
        updateRequest.setSuperviseDepartName(request.getSuperviseDepartName());
        this.organizationRpcProvider.update(updateRequest);
        return this.ok();
    }

    @ApiOperation(value = "判断企业信息是否完善", response = String.class)
    @PostMapping("/{organizationId}/info/complete")
    public ResponseEntity<CompanyInfoCompleteResponse> infoComplete(@PathVariable String organizationId) {
        return this.ok(organizationRpcProvider.infoComplete(organizationId));
    }
}
