package com.bcxin.identity.api.components.impls;

import com.bcxin.Infrastructures.components.JsonProvider;
import com.bcxin.Infrastructures.exceptions.BadTenantException;
import com.bcxin.api.interfaces.identities.responses.ThirdPartyCommonTokenResponse;
import com.bcxin.identity.api.components.WechatCgiProvider;
import com.bcxin.identity.api.components.responses.JsCode2SessionResponse;
import com.bcxin.identity.api.configs.WechatCgiConfig;
import org.apache.http.HttpResponse;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

@Component
public class WechatCgiProviderImpl implements WechatCgiProvider {
    private static final Logger logger = LoggerFactory.getLogger(WechatCgiProviderImpl.class);
    private static final String JS_CODE_2_SESSION_URL = "https://api.weixin.qq.com/sns/oauth2/access_token";
    private static final String WECHAT_ACCESS_TOKEN_URL = "https://api.weixin.qq.com/cgi-bin/token";

    private static final CloseableHttpClient httpclient;
    static {
        // 创建连接池管理器
        PoolingHttpClientConnectionManager connManager = new PoolingHttpClientConnectionManager();
        // 设置最大连接数
        connManager.setMaxTotal(500); // 总连接数
        connManager.setDefaultMaxPerRoute(80); // 每个路由的最大连接数

        // 设置请求配置
        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectTimeout(10000) // 连接超时时间（毫秒）
                .setSocketTimeout(10000) // 读取超时时间（毫秒）
                .build();

        // 创建 HttpClient 实例
        httpclient = HttpClients.custom()
                .setConnectionManager(connManager)
                .setDefaultRequestConfig(requestConfig)
                .build();
    }
    private final WechatCgiConfig cgiConfig;
    private final JsonProvider jsonProvider;

    public WechatCgiProviderImpl(WechatCgiConfig cgiConfig, JsonProvider jsonProvider) {
        this.cgiConfig = cgiConfig;
        this.jsonProvider = jsonProvider;
    }

    @Override
    public JsCode2SessionResponse execJsCode(String code) {
        String targetUrl = String.format(
                "%s?appid=%s&secret=%s&code=%s&grant_type=authorization_code",
                JS_CODE_2_SESSION_URL, cgiConfig.getAppid(),
                cgiConfig.getSecret(), code);

        String responseBody = null;
        try {
            HttpGet httpGet = new HttpGet(targetUrl);

            HttpResponse response = httpclient.execute(httpGet);
            responseBody = EntityUtils.toString(response.getEntity());

            /**
             * 比如: 这样的错误
             * {"errcode":40029,"errmsg":"invalid code, rid: 67a70694-50e86eef-57905837"}
             */
            if (responseBody.contains("errcode")) {
                throw new BadTenantException(responseBody);
            }

            JsCode2SessionResponse rs = this.jsonProvider.toObject(JsCode2SessionResponse.class, responseBody);

            return rs;
        } catch (Exception ex) {
            logger.error("do.execJsCode 异常:{};", targetUrl, ex);
            throw new BadTenantException(String.format("调用赛演微信jscode2session异常:code=%s;response=%s", code, responseBody), ex);
        }
    }

    @Override
    public ThirdPartyCommonTokenResponse getAccessToken() {
        String targetUrl = String.format(
                "%s?grant_type=client_credential&appid=%s&secret=%s",
                WECHAT_ACCESS_TOKEN_URL, cgiConfig.getAppid(),
                cgiConfig.getSecret());

        String responseBody = null;
        try {
            HttpGet httpGet = new HttpGet(targetUrl);
            HttpResponse response = httpclient.execute(httpGet);
            responseBody = EntityUtils.toString(response.getEntity());

            /**
             * 比如: 这样的错误
             * {"access_token":"90_822dSSnccPgKVSgs0lfXenFT34Fa7ovb78KOe3RSo-0h6eMjIOoy4rbtg4ZrfB1RT4MybKEXiJ7iecYqJmuIEzIBNg--GCMVUoTsDic6LfMIKtAPdLynttinZZ8FLLgAAAZFF","expires_in":7200}
             */
            if (!responseBody.contains("access_token")) {
                throw new BadTenantException(responseBody);
            }

            ThirdPartyCommonTokenResponse rs = this.jsonProvider.toObject(ThirdPartyCommonTokenResponse.class, responseBody);

            return rs;
        } catch (Exception ex) {
            logger.error("do.getAccessToken 异常:{};", targetUrl, ex);
            throw new BadTenantException(String.format("获取AccessToken:response=%s", responseBody), ex);
        }
    }

}
