package com.bcxin.tenant.apis.impls;


import com.bcxin.Infrastructures.enums.RelationshipsStatus;
import com.bcxin.Infrastructures.exceptions.ForbidTenantException;
import com.bcxin.api.interfaces.ApiConstant;
import com.bcxin.api.interfaces.tenants.OrganizationRelationshipRpcProvider;
import com.bcxin.api.interfaces.tenants.requests.organizationRelationship.OrganizationRelationshipCreateRequest;
import com.bcxin.api.interfaces.tenants.requests.organizationRelationship.OrganizationRelationshipInfoUpdateRequest;
import com.bcxin.api.interfaces.tenants.requests.organizationRelationship.OrganizationRelationshipStatusUpdateRequest;
import com.bcxin.api.interfaces.tenants.requests.organizationRelationship.OrganizationStructureQueryRequest;
import com.bcxin.api.interfaces.tenants.responses.OrgRelationResponse;
import com.bcxin.api.interfaces.tenants.responses.OrganizationRelationshipResponse;
import com.bcxin.api.interfaces.tenants.responses.OrganizationStructureResponse;
import com.bcxin.api.interfaces.tenants.responses.OrganizationTypeResponse;
import com.bcxin.tenant.apis.translates.OrganizationRelationshipTranslate;

import com.bcxin.tenant.domain.repositories.OrganizationRelationshipRepository;
import com.bcxin.tenant.domain.repositories.dtos.OrgRelationDto;
import com.bcxin.tenant.domain.repositories.dtos.OrganizationRelationshipInfoDto;
import com.bcxin.tenant.domain.repositories.dtos.OrganizationStatisticsDto;
import com.bcxin.tenant.domain.repositories.dtos.OrganizationTypeDto;
import com.bcxin.tenant.domain.services.OrganizationRelationshipService;
import com.bcxin.tenant.domain.services.commands.organizationrelationship.OrganizationRelationshipInfoUpdateCommand;
import com.google.common.base.Functions;
import io.jsonwebtoken.lang.Collections;
import org.apache.commons.lang3.StringUtils;
import org.apache.dubbo.config.annotation.DubboService;
import org.springframework.util.CollectionUtils;

import java.sql.Timestamp;
import java.util.*;
import java.util.stream.Collectors;

@DubboService(version = ApiConstant.VERSION,validation = "true",retries = 0,timeout = 120_000)
public class OrganizationRelationshipRpcProviderImpl  implements OrganizationRelationshipRpcProvider {

    private final OrganizationRelationshipService organizationRelationshipService;
    private final OrganizationRelationshipRepository organizationRelationshipRepository;

    private final OrganizationRelationshipTranslate organizationRelationshipTranslate;

    public OrganizationRelationshipRpcProviderImpl(OrganizationRelationshipService organizationRelationshipService, OrganizationRelationshipRepository organizationRelationshipRepository, OrganizationRelationshipTranslate organizationRelationshipTranslate) {
        this.organizationRelationshipService = organizationRelationshipService;
        this.organizationRelationshipRepository = organizationRelationshipRepository;
        this.organizationRelationshipTranslate = organizationRelationshipTranslate;
    }

    @Override
    public void join(OrganizationRelationshipCreateRequest organizationRelationshipCreateRequest) {
        organizationRelationshipService.join(organizationRelationshipCreateRequest);
    }

    @Override
    public OrganizationRelationshipResponse findRelationshipDetailById(long id) {
        OrganizationRelationshipInfoDto byId = organizationRelationshipRepository.getRelationshipDetailById(id);
        if (byId == null) {
            throw new ForbidTenantException("组织关系不存在");
        }
        OrganizationTypeDto organizationType=byId.getOrganizationType();
        OrganizationStatisticsDto organizationStatistics = byId.getOrganizationStatistics();
            return OrganizationRelationshipResponse.create(
                    byId.getId(),
                    byId.getParentId(),
                    byId.getParentName(),
                    byId.getSelectedOrganizationId(),
                    byId.getSelectedOrganizationName(),
                    byId.getCode(),
                    byId.getContactName(),
                    byId.getContactTelephone(),
                    organizationType ==null? null:OrganizationTypeResponse.create(
                            organizationType.getId(),
                            organizationType.getName(),
                            organizationType.getNote(),
                            organizationType.getOrganizationId(),
                            null,null,null,null),
                    byId.getRelationshipIndexTree(),
                    byId.getStatus().ordinal(),
                    byId.getLastReplyNote(),
                    byId.getLastReplyEmployeeId(),
                    byId.getLastReplyEmployeeName(),
                    byId.getLastReplyTime()==null?null:new Timestamp(byId.getLastReplyTime().getTime()),
                    byId.getOrganizationId(),
                    null,
                    null,
                    null,
                    byId.getInviteType(),
                    organizationStatistics == null ? null : organizationStatistics.getUnifySocialCreditCode(),
                    organizationStatistics == null ? 0 : organizationStatistics.getCountOfMembers()
            );
    }

    @Override
    public OrganizationStructureResponse getOrganizationStructure(OrganizationStructureQueryRequest request) {
        Collection<RelationshipsStatus> statuses = request.getStatuses();
        String groupOrgId = request.getOrganizationID();
        if(CollectionUtils.isEmpty(statuses)){
            statuses = Arrays.stream(RelationshipsStatus.values()).collect(Collectors.toList());
        }

        Collection<OrganizationRelationshipInfoDto> organizationInfoDtos = organizationRelationshipRepository
                .getOrganizationRelationshipsInfo(groupOrgId, statuses);

        if(Collections.isEmpty(organizationInfoDtos)){
            return null;
        }

        OrganizationStructureResponse treeRoot = null;
        Map<String, OrganizationStructureResponse> orgMap = organizationInfoDtos.stream()
                .map(ix-> organizationRelationshipTranslate.translate2OrganizationStructureResponse(ix))
                .collect(Collectors.toMap(OrganizationStructureResponse::getSelectedOrganizationId, Functions.identity()));

        for(Iterator<OrganizationRelationshipInfoDto> itr = organizationInfoDtos.iterator(); itr.hasNext();){
            OrganizationRelationshipInfoDto orgInfo = itr.next();
            if(StringUtils.isEmpty(orgInfo.getParentId())){
                treeRoot = orgMap.get(orgInfo.getSelectedOrganizationId());
            }else{
                OrganizationStructureResponse parentResponse = orgMap.get(orgInfo.getParentId());
                if(parentResponse != null){
                    parentResponse.getChildren().add(orgMap.get(orgInfo.getSelectedOrganizationId()));
                }
            }
        }
        return treeRoot;
    }

    @Override
    public void updateStatus(String organizationId,OrganizationRelationshipStatusUpdateRequest organizationRelationshipStatusUpdateRequest) {

        organizationRelationshipService.updateOrganizationRelationshipStatus(organizationId,organizationRelationshipStatusUpdateRequest);

    }

    @Override
    public void updateRelationshipInfo(OrganizationRelationshipInfoUpdateRequest organizationRelationshipUpdateRequest) {
        OrganizationRelationshipInfoUpdateCommand command = organizationRelationshipTranslate.translate2InfoUpdateCommand(organizationRelationshipUpdateRequest);
        organizationRelationshipService.updateOrganizationRelationshipInfo(command);
    }

    @Override
    public void delete(Collection<Long> ids,String organizationId) {
        organizationRelationshipService.delete(ids,organizationId);
    }

    @Override
    public List<OrgRelationResponse> getParentList(String organizationId) {
        Collection<OrgRelationDto> parentList = organizationRelationshipRepository.getParentList(organizationId);
        List<OrgRelationResponse> response = null;
        if(!CollectionUtils.isEmpty(parentList)){
            response = parentList.stream()
                    .map(ix-> OrgRelationResponse.create(ix.getRelationId(),
                            ix.getOrgName(),
                            ix.getUnifySocialCreditCode()))
                    .collect(Collectors.toList());
        }
        return response;
    }

    @Override
    public List<OrgRelationResponse> getChildrenList(String organizationId) {
        Collection<OrgRelationDto> childrenList = organizationRelationshipRepository.getChildrenList(organizationId);
        List<OrgRelationResponse> response = null;
        if(!CollectionUtils.isEmpty(childrenList)){
            response = childrenList.stream()
                    .map(ix-> OrgRelationResponse.create(ix.getRelationId(),
                            ix.getOrgName(),
                            ix.getUnifySocialCreditCode()))
                    .collect(Collectors.toList());
        }
        return response;
    }
}
