package com.bcxin.tenant.apis.impls;

import com.alibaba.excel.EasyExcel;
import com.bcxin.Infrastructures.Pageable;
import com.bcxin.Infrastructures.exceptions.NotFoundTenantException;
import com.bcxin.api.interfaces.ApiConstant;
import com.bcxin.api.interfaces.commons.CommonImportResponse;
import com.bcxin.api.interfaces.tenants.ContractRpcProvider;
import com.bcxin.api.interfaces.tenants.criterias.ContractCriteria;
import com.bcxin.api.interfaces.tenants.requests.contracts.BatchDeleteContractRequest;
import com.bcxin.api.interfaces.tenants.requests.contracts.BatchImportContractRequest;
import com.bcxin.api.interfaces.tenants.requests.contracts.CreateContractRequest;
import com.bcxin.api.interfaces.tenants.requests.contracts.UpdateContractRequest;
import com.bcxin.api.interfaces.tenants.responses.ContractGetResponse;
import com.bcxin.api.interfaces.tenants.responses.ContractReportResponse;
import com.bcxin.tenant.domain.configs.PlatformUploadConfig;
import com.bcxin.tenant.domain.entities.ContractEntity;
import com.bcxin.tenant.domain.readers.TenantDbReader;
import com.bcxin.tenant.domain.readers.criterias.EmployeeContractCriteria;
import com.bcxin.tenant.domain.readers.dtos.EmployeeContractDto;
import com.bcxin.tenant.domain.repositories.ContractRepository;
import com.bcxin.tenant.domain.repositories.dtos.ContractExportDto;
import com.bcxin.tenant.domain.repositories.dtos.ContractReportDto;
import com.bcxin.tenant.domain.services.ContractService;
import com.bcxin.tenant.domain.services.commands.contracts.*;
import com.bcxin.tenant.domain.services.commands.contracts.results.BatchImportContractCommandResult;
import org.apache.dubbo.config.annotation.DubboService;

import java.io.File;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

@DubboService(version = ApiConstant.VERSION,validation = "true",timeout = 120 *1000,retries = 0)
public class ContractRpcProviderImpl implements ContractRpcProvider {
    private final ContractService contractService;
    private final TenantDbReader dbReader;
    private final ContractRepository contractRepository;

    public ContractRpcProviderImpl(ContractService contractService, TenantDbReader dbReader, ContractRepository contractRepository) {
        this.contractService = contractService;
        this.dbReader = dbReader;
        this.contractRepository = contractRepository;
    }

    @Override
    public void create(String organizationId, CreateContractRequest request) {

        this.contractService.dispatch(CreateContractCommand.create(
                request.getName(),
                request.getEmployeeId(), organizationId,
                request.getAName(), request.getBName(), request.getBeginDate(),
                request.getEndDate(), request.isDateLimitless(),
                request.getAttachment(), request.getNote()));
    }

    @Override
    public void update(String organizationId, String id, UpdateContractRequest request) {
        this.contractService.dispatch(
                UpdateContractCommand.create(
                        organizationId, id, request.getName(), request.getAName(), request.getBName(),
                        request.getBeginDate(), request.getEndDate(),
                        request.isDateLimitless(),
                        request.getAttachment(),
                        request.getNote()));
    }

    @Override
    public void delete(String organizationId, String id) {
        this.contractService.dispatch(DeleteContractCommand.create(organizationId, id));
    }

    @Override
    public void batchDelete(String organizationId, BatchDeleteContractRequest request) {
        this.contractService.dispatch(BatchDeleteContractCommand.create(organizationId, request.getIds()));
    }

    @Override
    public Pageable<ContractGetResponse> find(String organizationId, ContractCriteria criteria) {
        Pageable<EmployeeContractDto> employeeContractDtoPageable =
                this.dbReader.findEmployeeContracts(organizationId,
                        EmployeeContractCriteria.create(
                                criteria.getPageIndex(),
                                criteria.getPageSize(),
                                criteria.getKeyword(),
                                criteria.getEmployeeId(),
                                criteria.getDepartIds(),
                                criteria.getOccupationTypes(),
                                criteria.getEmploymentStatus(),
                                criteria.getBeginDate(),
                                criteria.getEndDate(),
                                criteria.getStatus(),
                                criteria.getBeginCreatedTime(),
                                criteria.getEndCreatedTime()));

        return Pageable.create(
                employeeContractDtoPageable.getPageIndex(),
                employeeContractDtoPageable.getPageSize(),
                employeeContractDtoPageable.getTotalCount(),
                employeeContractDtoPageable.getData()
                        .stream().map(ix -> ContractGetResponse.create(
                                ix.getId(),
                                ix.getName(),
                                ix.getIdNum(),
                                ix.getCredentialType(),
                                ix.getAName(),
                                ix.getBName(),
                                ix.getEmployeeId(),
                                ix.getEmployeeName(),
                                ix.getEmployeeTelephone(),
                                ix.getEmployeeDepartName(),
                                ix.getEmployeeOccupationType(),
                                ix.getEmployeeStatus(),
                                ix.getBeginDate(),
                                ix.getEndDate(),
                                ix.getDateLimitless(),
                                ix.getAttachment(),
                                ix.getNote(),
                                ix.getStatus(),
                                ix.getLastModifier() != null ?ix.getLastModifier().getName():null,
                                ix.getLastModifier() != null ?ix.getLastModifier().getOperatorTime():null,ix.getContractStatus(),
                                ix.getCreator().getCreatedTime()))
                        .collect(Collectors.toList()));
    }

    @Override
    public CommonImportResponse batchImport(String organizationId, String path,
                                            Collection<BatchImportContractRequest> requests) {
        AtomicInteger paramIndexAto = new AtomicInteger(0);
        Collection<BatchImportContractCommand.ContractCommandItem> commandItems =
                requests.stream().map(ii -> {
                            paramIndexAto.incrementAndGet();
                            return BatchImportContractCommand.ContractCommandItem.create(
                                    paramIndexAto.get(),
                                    ii.getName(), ii.getAName(),
                                    ii.getBName(), ii.getIdNum(), ii.getBeginDate(),
                                    ii.getEndDate());
                        }
                ).collect(Collectors.toList());
        BatchImportContractCommand command = BatchImportContractCommand.create(organizationId, path, commandItems);

        BatchImportContractCommandResult result = this.contractService.dispatch(command);

        return CommonImportResponse.create(
                result.getResourceId(),
                result.getTotalCount(),
                result.getTotalSuccess(),
                result.getTotalFailed(),
                "执行完毕");
    }

    @Override
    public String exportExcel(String organizationId, ContractCriteria criteria) {
        List<ContractExportDto> list = this.dbReader.findContractList(organizationId,criteria);
        String path = PlatformUploadConfig.getDownloadPath()+"合同数据"+ Math.random()+".xlsx";
        File file = new File(path);
        if(!file.getParentFile().exists()){
            file.mkdirs();
        }
        EasyExcel.write(path,ContractExportDto.class).sheet().doWrite(list);
        return path;
    }

    @Override
    public ContractReportResponse findReport(String organizationId, ContractCriteria criteria) {
        ContractReportDto contractReportDto = this.dbReader.findContractReport(organizationId,criteria);
        return ContractReportResponse.create(contractReportDto.getTotal(),
                contractReportDto.getIncompleteCount(),
                contractReportDto.getEffectCount(),
                contractReportDto.getBeEffectCount(),
                contractReportDto.getDueCount(),
                contractReportDto.getStopCount(),
                contractReportDto.getUnSignCount(),
                contractReportDto.getSecUnSignCount(),
                contractReportDto.getLeaveCount());
    }

    @Override
    public String  getOrganIdAndId(String organizationId, String id) {
        ContractEntity contractEntity =contractRepository.getByOrganIdAndId(organizationId,id);
        if (contractEntity == null) {
            throw new NotFoundTenantException();
        }
       String attachment =contractEntity.getAttachment();
        return attachment;
    }


}
