package com.bcxin.tenant.apis.impls;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bcxin.Infrastructures.enums.*;
import com.bcxin.Infrastructures.utils.DictUtil;
import com.bcxin.Infrastructures.utils.RedisUtil;
import com.bcxin.api.interfaces.ApiConstant;
import com.bcxin.api.interfaces.commons.ConfigDictRpcProvider;
import com.bcxin.tenant.domain.entities.ConfigDictEntity;
import com.bcxin.tenant.domain.entities.SysIndustryTypeEntity;
import com.bcxin.tenant.domain.entities.SysInstitutionalTypeEntity;
import com.bcxin.tenant.domain.repositories.ConfigDictRepository;
import com.bcxin.tenant.domain.repositories.SysIndustryTypeRepository;
import com.bcxin.tenant.domain.repositories.SysInstitutionalTypeRepository;
import org.apache.commons.lang3.StringUtils;
import org.apache.dubbo.config.annotation.DubboService;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * <b> 字典 </b>
 * @author ZXF
 * @create 2021/12/21 0021 11:06
 * @version
 * @注意事项 </b>
 */
@DubboService(version = ApiConstant.VERSION,validation = "true",timeout = 120 *1000,retries = 0)
public class ConfigDictRpcProviderImpl implements ConfigDictRpcProvider {


    private final ConfigDictRepository configDictRepository;

    private final RedisUtil redisUtil;

    private final SysIndustryTypeRepository sysIndustryTypeRepository;

    private final SysInstitutionalTypeRepository sysInstitutionalTypeRepository;
    private final NamedParameterJdbcTemplate jdbcTemplate;

    private final static String DICT_REDIS_KEY_ALL = "com:bcx:dict:all";
    private final static String REGISTER_DICT_REDIS_KEY = "com:bcx:register:dict";
    private final static String REDIS_KEY_INDUSTRY_CACHE_KEY = "com:bcx:v2:industry";
    private final static String REDIS_KEY_INSTITUTIONAL_CACHE_KEY = "com:bcx:v2:institutional:";

    public ConfigDictRpcProviderImpl(ConfigDictRepository configDictRepository, RedisUtil redisUtil,
                                     SysIndustryTypeRepository sysIndustryTypeRepository,
                                     SysInstitutionalTypeRepository sysInstitutionalTypeRepository, NamedParameterJdbcTemplate jdbcTemplate) {
        this.configDictRepository = configDictRepository;
        this.redisUtil = redisUtil;
        this.sysIndustryTypeRepository = sysIndustryTypeRepository;
        this.sysInstitutionalTypeRepository = sysInstitutionalTypeRepository;
        this.jdbcTemplate = jdbcTemplate;
    }

    @Override
    public JSONObject findAll() {
        Object redisCache = redisUtil.get(DICT_REDIS_KEY_ALL);
        if (redisCache != null) {
            return JSONObject.parseObject(redisCache.toString(), JSONObject.class);
        }

        Collection<ConfigDictEntity> dictList = configDictRepository.findAll();
        JSONObject json = new JSONObject();
        if (dictList.size() > 0) {
            String codeType = "";
            JSONObject lJson;
            JSONArray jArr = null;
            for (ConfigDictEntity entity : dictList) {
                if (jArr == null) {
                    jArr = new JSONArray();
                    codeType = entity.getCodeType();
                }
                if (!entity.getCodeType().equals(codeType)) {
                    json.put(codeType, jArr);
                    jArr = new JSONArray();
                    codeType = entity.getCodeType();
                }
                lJson = new JSONObject();
                lJson.put("label", entity.getLabel());
                lJson.put("value", entity.getCodeValue());
                jArr.add(lJson);
            }
            //最后一个只能在完成循环后插入
            json.put(codeType, jArr);
        }

        redisUtil.set(DICT_REDIS_KEY_ALL, JSONObject.toJSONString(json), 10 * 60 * 60);

        return json;
    }

    /**
     * description: 根据行业类型，获取机构类型列表
     * author: linchunpeng
     * date:  2023-05-08 13:10
     */
    @Override
    public List<Map<String, String>> getOrgTypeList(String industryTypeValue) {
        if (StringUtils.isNotBlank(industryTypeValue)) {
            // 根据行业编码查询行业类型
            SysIndustryTypeEntity industryType = sysIndustryTypeRepository.findByCodeAndIsDeleted(industryTypeValue, 0);
            if (industryType == null) {
                return new ArrayList<>();
            }

            // 根据行业ID查询机构类型列表
            Collection<SysInstitutionalTypeEntity> institutionalTypes =
                    sysInstitutionalTypeRepository.findByIndustryIdAndIsDeleted(industryType.getId(), 0);

            List<Map<String, String>> orgTypeList = new ArrayList<>();
            if (institutionalTypes != null && !institutionalTypes.isEmpty()) {
                for (SysInstitutionalTypeEntity entity : institutionalTypes) {
                    orgTypeList.add(DictUtil.getDictMap(entity.getCode(), entity.getName()));
                }
            }
            return orgTypeList;
        }
        return new ArrayList<>();
    }

    /**
     * description: 获取注册页面数据字典列表
     * author: linchunpeng
     * date:  2023-05-08 14:01
     */
    @Override
    public Map<String, List<Map<String, String>>> findRegisterDict() {
        Map<String, List<Map<String, String>>> resultMap = new HashMap<>();

        Object redisCache = redisUtil.get(REGISTER_DICT_REDIS_KEY);
        if (redisCache != null) {
            return JSONObject.parseObject(redisCache.toString(), Map.class);
        }

        //行业类型 - 从数据库查询
        List<Map<String, String>> industryTypeList = new ArrayList<>();
        Collection<SysIndustryTypeEntity> industryTypes = sysIndustryTypeRepository.findByIsDeleted(0);
        if (industryTypes != null && !industryTypes.isEmpty()) {
            for (SysIndustryTypeEntity entity : industryTypes) {
                industryTypeList.add(DictUtil.getDictMap(entity.getCode(), entity.getName()));
            }
        }
        resultMap.put("industryType", industryTypeList);

        //保安服务公司类型
        List<Map<String, String>> industryDetailTypeList = new ArrayList<>();
        for (IndustryDetailType value : IndustryDetailType.values()) {
            if (value.ordinal() > 0) {
                industryDetailTypeList.add(DictUtil.getDictMap("" + value.ordinal(), value.getTypeName()));
            }
        }
        resultMap.put("industryDetailType", industryDetailTypeList);

        //总分子公司
        List<Map<String, String>> parentSubsidiaryList = new ArrayList<>();
        for (ParentSubsidiary value : ParentSubsidiary.values()) {
            if (value.ordinal() > 0) {
                parentSubsidiaryList.add(DictUtil.getDictMap("" + value.ordinal(), value.getTypeName()));
            }
        }
        resultMap.put("parentSubsidiary", parentSubsidiaryList);

        //经济类型
        List<Map<String, String>> economicTypeList = new ArrayList<>();
        for (EconomicType value : EconomicType.values()) {
            if (value.ordinal() > 0) {
                economicTypeList.add(DictUtil.getDictMap("" + value.ordinal(), value.getTypeName()));
            }
        }
        resultMap.put("economicType", economicTypeList);

        //公司类型
        List<Map<String, String>> companyNatureList = new ArrayList<>();
        for (CompanyNature value : CompanyNature.values()) {
            if (value.ordinal() > 0) {
                companyNatureList.add(DictUtil.getDictMap("" + value.ordinal(), value.getTypeName()));
            }
        }
        resultMap.put("companyNature", companyNatureList);

        //公司证件类型
        List<Map<String, String>> companyCertificateTypeList = new ArrayList<>();
        for (CompanyCertificateType value : CompanyCertificateType.values()) {
            companyCertificateTypeList.add(DictUtil.getDictMap("" + value.ordinal(), value.getTypeName()));
        }
        resultMap.put("companyCertificateType", companyCertificateTypeList);

        //人员证件类型
        List<Map<String, String>> credentialTypeList = new ArrayList<>();
        for (CredentialType value : CredentialType.values()) {
            if (value.isIdentityNo()) {
                credentialTypeList.add(DictUtil.getDictMap("" + value.ordinal(), value.getTypeName()));
            }
        }
        resultMap.put("credentialType", credentialTypeList);

        //服务范围
        List<Map<String, String>> serviceScopeList = new ArrayList<>();
        for (ServiceScope value : ServiceScope.values()) {
            if (value.ordinal() > 0) {
                serviceScopeList.add(DictUtil.getDictMap(value.getTypeValue(), value.getTypeName()));
            }
        }
        resultMap.put("serviceScope", serviceScopeList);

        //人力防范评定等级
        List<Map<String, String>> securityPreparednessRatingList = new ArrayList<>();
        for (SecurityPreparednessRating value : SecurityPreparednessRating.values()) {
            securityPreparednessRatingList.add(DictUtil.getDictMap("" + value.ordinal(), value.getTypeName()));
        }
        resultMap.put("securityPreparednessRating", securityPreparednessRatingList);

        //武装守护押运评定等级
        List<Map<String, String>> securityArmedRatingList = new ArrayList<>();
        for (SecurityArmedRating value : SecurityArmedRating.values()) {
            securityArmedRatingList.add(DictUtil.getDictMap("" + value.ordinal(), value.getTypeName()));
        }
        resultMap.put("securityArmedRating", securityArmedRatingList);

        //安全技术防范评定等级
        List<Map<String, String>> securitySafetyDefenseRatingList = new ArrayList<>();
        for (SecuritySafetyDefenseRating value : SecuritySafetyDefenseRating.values()) {
            securitySafetyDefenseRatingList.add(DictUtil.getDictMap("" + value.ordinal(), value.getTypeName()));
        }
        resultMap.put("securitySafetyDefenseRating", securitySafetyDefenseRatingList);

        //自招保安单位性质
        resultMap.put("natureOfSelfRecruitedUnits", NatureOfSelfRecruitedUnits.getDictList());

        //国籍
        resultMap.put("nationality", Nationality.getDictList());

        //放入缓存
        redisUtil.set(REGISTER_DICT_REDIS_KEY, JSONObject.toJSONString(resultMap), -1);
        return resultMap;
    }

    /**
     * description: 删除注册页面数据字典缓存
     * author: linchunpeng
     * date:  2023-05-08 14:34
     */
    @Override
    public void clearRegisterDictRedis() {
        redisUtil.del(REGISTER_DICT_REDIS_KEY);
        redisUtil.del(REDIS_KEY_INDUSTRY_CACHE_KEY);
        Set<String> keys = redisUtil.keys(REDIS_KEY_INSTITUTIONAL_CACHE_KEY + "*");
        keys.forEach(redisUtil::del);
    }

    @Override
    public List<Map<String, String>> getIndustryList() {
        List<Map<String, String>> map = (List<Map<String, String>>) redisUtil.get(REDIS_KEY_INDUSTRY_CACHE_KEY);
        if (map == null) {
            map =
                    this.jdbcTemplate.query("select id,`name` from sys_industry_types",
                            new HashMap<>(), (rs, rowNum) -> {
                                Map<String, String> mp = new HashMap<>();
                                mp.put("id", rs.getString("id"));
                                mp.put("name", rs.getString("name"));
                                return mp;
                            });

            redisUtil.set(REDIS_KEY_INDUSTRY_CACHE_KEY, map, 10 * 60 * 60);
        }

        return map;
    }

    @Override
    public List<Map<String, String>> getInstitutionalListByIndustryId(String id) {
        String institutionalCacheKey = String.format(REDIS_KEY_INSTITUTIONAL_CACHE_KEY, id);
        List<Map<String, String>> map = (List<Map<String, String>>) redisUtil.get(institutionalCacheKey);
        if (map == null) {
            Map<String, String> params = new HashMap<>();
            params.put("industry_id", id);
            map = this.jdbcTemplate.query("select id,`name` from sys_institutional_types where industry_id=:industry_id",
                    params, (rs, rowNum) -> {
                        Map<String, String> mp = new HashMap<>();
                        mp.put("id", rs.getString("id"));
                        mp.put("name", rs.getString("name"));
                        return mp;
                    });

            redisUtil.set(institutionalCacheKey, map, 10 * 60 * 60);
        }

        return map;
    }

    @Override
    public List<Map<String, String>> getInstitutionalListByIds(Collection<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return Collections.EMPTY_LIST;
        }

        Map<String, Object> params = new HashMap<>();
        params.put("ids", ids);
        List<Map<String, String>> map = this.jdbcTemplate.query("select id,`name` from sys_institutional_types where id in (:ids)",
                params, (rs, rowNum) -> {
                    Map<String, String> mp = new HashMap<>();
                    mp.put("id", rs.getString("id"));
                    mp.put("name", rs.getString("name"));
                    return mp;
                });

        return map;
    }
}
