package com.bcxin.identity.api.impls;

import cn.hutool.core.util.StrUtil;
import com.bcxin.Infrastructures.SystemConstant;
import com.bcxin.Infrastructures.TenantContext;
import com.bcxin.Infrastructures.TenantUserContext;
import com.bcxin.Infrastructures.components.JsonProvider;
import com.bcxin.Infrastructures.exceptions.AccountLockedException;
import com.bcxin.Infrastructures.exceptions.BadTenantException;
import com.bcxin.Infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.Infrastructures.utils.JwtUtil;
import com.bcxin.api.interfaces.ApiConstant;
import com.bcxin.api.interfaces.identities.IdentityRpcProvider;
import com.bcxin.api.interfaces.identities.requests.*;
import com.bcxin.api.interfaces.identities.responses.PrepareResetPasswordResponse;
import com.bcxin.api.interfaces.identities.responses.SignInResponse;
import com.bcxin.api.interfaces.identities.responses.ThirdPartyCommonTokenResponse;
import com.bcxin.api.interfaces.identities.responses.ToCWechatCgiBinSignInResponse;
import com.bcxin.api.interfaces.identities.valueTypes.WechatCgiBinSubject;
import com.bcxin.api.interfaces.tenants.requests.employees.ResetPassWordRequest;
import com.bcxin.identity.api.components.WechatCgiProvider;
import com.bcxin.identity.api.components.responses.JsCode2SessionResponse;
import com.bcxin.identity.api.utils.RSAUtils;
import com.bcxin.identity.domains.components.PasswordEncoder;
import com.bcxin.identity.domains.components.SmsExecutor;
import com.bcxin.identity.domains.entities.IdentityUserEntity;
import com.bcxin.identity.domains.entities.UserNamePasswordPrincipalEntity;
import com.bcxin.identity.domains.enums.ForgetPasswordAction;
import com.bcxin.identity.domains.exceptions.IdentityNotFoundException;
import com.bcxin.identity.domains.repositories.IdentityUserRepository;
import com.bcxin.identity.domains.services.IdentityUserService;
import com.bcxin.identity.domains.services.commandResult.PrepareResetPasswordCommandResult;
import com.bcxin.identity.domains.services.commandResult.WechatCgiBinSignInCommandResult;
import com.bcxin.identity.domains.services.commands.*;
import org.apache.dubbo.config.annotation.DubboService;
import org.apache.dubbo.rpc.RpcContext;

import java.util.*;
import java.util.stream.Collectors;

@DubboService(version = ApiConstant.VERSION,validation = "true",timeout = 120 *1000,retries = 0)
public class IdentityRpcProviderImpl implements IdentityRpcProvider {
    private final IdentityUserService identityUserService;
    private final IdentityUserRepository identityUserRepository;
    private final PasswordEncoder passwordEncoder;
    private final SmsExecutor smsExecutor;

    private final WechatCgiProvider wechatCgiProvider;

    private final JsonProvider jsonProvider;

    public IdentityRpcProviderImpl(IdentityUserService identityUserService,
                                   IdentityUserRepository identityUserRepository,
                                   SmsExecutor smsExecutor, PasswordEncoder passwordEncoder,
                                   WechatCgiProvider wechatCgiProvider, JsonProvider jsonProvider) {
        this.identityUserService = identityUserService;
        this.identityUserRepository = identityUserRepository;
        this.smsExecutor = smsExecutor;
        this.passwordEncoder = passwordEncoder;
        this.wechatCgiProvider = wechatCgiProvider;
        this.jsonProvider = jsonProvider;
    }

    @Override
    public String getByName() {
        StringBuilder sb = new StringBuilder();
        RpcContext rpcContext = RpcContext.getServerContext();

        sb.append(String.format("%s:", new Date()));
        sb.append(String.format("URL=%s;", rpcContext.getConsumerUrl()));
        sb.append(String.format("注册信息:%s:%s:%s;", rpcContext.getProtocol(), rpcContext.getRemoteHost(), rpcContext.getRemotePort()));
        if (rpcContext.getArguments() != null) {
            Arrays.stream(rpcContext.getArguments()).forEach(arg -> {
                sb.append(arg);
            });
        }

        return sb.toString();
    }

    @Override
    public void changePassword(ChangePasswordRequest request) {
        ChangePasswordCommand command =
                ChangePasswordCommand.create(
                        request.getIdentityUserId(),
                        request.getUserName(),
                        formatFrontPassword(request.isRsa(), request.getPassword()),
                        formatFrontPassword(request.isRsa(), request.getNewPassword()),
                        formatFrontPassword(request.isRsa(), request.getConfirmedPassword())
                );
        if (!isValidPassword(command.getNewPassword())){
            throw new BadTenantException("密码长度至少8位，由数字、大小写字母和特殊字符混合组成。");
        }
        this.identityUserService.changePassword(command);
    }

//    private static boolean isValidPassword(String password) {
//        // 正则表达式说明：
//        // ^ 表示字符串开始
//        // (?=.*[0-9]) 表示至少有一个数字
//        // (?=.*[a-z]) 表示至少有一个小写字母
//        // (?=.*[A-Z]) 表示至少有一个大写字母
//        // (?=.*[@#$%^&+=]) 表示至少有一个特殊字符
//        // .{8,} 表示字符串长度至少为8
//        String passwordPattern = "^(?=.*[0-9])" +
//                "(?=.*[a-z])" +
//                "(?=.*[A-Z])" +
//                "(?=.*[@#$%^&+=])" +
//                "(?=\\S+$).{8,}";
//        return password.matches(passwordPattern);
//    }

    private boolean isValidPassword(String password) {
        // 正则表达式规则，密码必须包含至少一个小写字母、至少一个大写字母、至少一个数字和至少一个特殊字符、长度至少为8
        String regex = "^(?=.*[a-z])(?=.*[A-Z])(?=.*\\d)(?=.*[\\\\!\"#$%&'()*+,\\-./:;<=>?@\\[\\]^_`{|}~])[A-Za-z\\d\\\\!\"#$%&'()*+,\\-./:;<=>?@\\[\\]^_`{|}~]{8,}$";
        return password.matches(regex);
    }

    @Override
    public void resetPassword(ResetPassWordRequest request){
        this.identityUserService.dispatch(ResetPasswordCommandAdmin.create(request.getLoginName(),request.getNewPassword()));
    }

    @Override
    public void bindWechat(BindWechatRequest request) {
        String tenantId = TenantContext.getInstance().getUserContext().get().getId();
        this.identityUserService.dispatch(WechatBindCommand.create(
                tenantId, request.getOpenId(), request.getNicky(), request.getUnionId()));
    }

    @Override
    public SignInResponse signIn(WeChatSignInRequest request) {
        WechatSignInCommand command = WechatSignInCommand.create(request.getOpenId(), request.getIpAddress());
        SignInCommand.SignInCommandResult result = this.identityUserService.dispatch(command);

        return SignInResponse.create(result.getToken(), result.getName(), result.getIdNum(), result.getUpdatePasswordType());
    }

    @Override
    public void change(ChangeBasicIdentityRequest request) {
        this.identityUserService.dispatch(UpdateIdentityBasicCommand.create(
                request.getTenantUserId(),
                request.getName(),
                request.getTelephone(),
                request.getIdNum()));
    }

    @Override
    public void unbindWechat(String id) {
        this.identityUserService.dispatch(WechatUnBindCommand.create(id));
    }

    /**
     * description：判断是否绑定微信unionid
     * author：linchunpeng
     * date：2023/10/12
     */
    @Override
    public boolean isBindUnionId(String id) {
        return this.identityUserService.dispatch(WechatCheckUnionIdCommand.create(id));
    }

    @Override
    public PrepareResetPasswordResponse prepareResetPassword(PrepareResetPasswordRequest request) {
        ForgetPasswordAction action = null;
        if (request.getActionType() != null) {
            switch (request.getActionType()) {
                case FaceIdentity:
                    action = ForgetPasswordAction.FaceIdentity;
                    break;
                case SMS:
                    action = ForgetPasswordAction.SMS;
                    break;
            }
        }

        PrepareResetPasswordCommandResult result =
                this.identityUserService.dispatch(
                        PrepareResetPasswordCommand.create(
                                request.getLoginName(),
                                request.getData(),
                                action
                        )
                );

        if (result == null) {
            return null;
        }

        return PrepareResetPasswordResponse.create(result.getJwt(), result.isSuccess(), result.getCode(),result.getMobile());
    }

    @Override
    public PrepareResetPasswordResponse prepareResetPasswordExist(PrepareResetPasswordRequest request) {

        PrepareResetPasswordCommandResult result =
                this.identityUserService.getUserMobile(
                        PrepareResetPasswordCommand.create(
                                request.getLoginName(),
                                request.getData(),
                                null
                        )
                );

        return PrepareResetPasswordResponse.create(result.getJwt(), result.isSuccess(), result.getCode(),result.getMobile(),result.isHasHeadPhoto());
    }

    @Override
    public String unlock(UnlockRequest request) {

        if (StrUtil.isEmpty(request.getUserid())||StrUtil.isEmpty(request.getPassword())) {
            throw new BadTenantException("参数不能为空");
        }

        IdentityUserEntity identityUserEntity =
                this.identityUserRepository.findByTenantUserId(request.getUserid())
                        .stream().findFirst().orElse(null);

        if (identityUserEntity == null) {
            throw new IdentityNotFoundException();
        }
        //对比密码
        Optional<UserNamePasswordPrincipalEntity> userNamePasswordOptional =
                identityUserEntity.getPrincipals().stream().filter(ii -> ii instanceof UserNamePasswordPrincipalEntity)
                        .map(ii -> (UserNamePasswordPrincipalEntity) ii)
                        .findFirst();
        if (userNamePasswordOptional.isPresent()) {
            UserNamePasswordPrincipalEntity principal = userNamePasswordOptional.get();
            //判断账号是否锁定30分钟
            if (principal.isLock()) {
                throw new AccountLockedException(String.format("您的登录密码已连续输错5次，账号已被安全锁定，请于%s分钟后重新登录。", principal.getLockRemainingTime()));
            }
            try {
                String password = formatFrontPassword(true, request.getPassword());
                if (passwordEncoder.isMatched(principal.getPassword(), password)) {
                    //登录成功，清除登录失败计数
                    identityUserService.loginFailClearCount(principal);
                    return JwtUtil.getToken(request.getUserid());
                } else {
                    throw new BadTenantException("密码错误！");
                }
            } catch (Exception e) {
                //登录失败增加计数
                int count = identityUserService.loginFailAddCount(principal);
                if (count >= 5) {
                    throw new AccountLockedException(String.format("您的登录密码已连续输错5次，账号已被安全锁定，请于%s分钟后重新登录。", principal.getLockRemainingTime()));
                }
                throw new BadTenantException("密码错误！");
            }
        }
        throw new IdentityNotFoundException();
    }

    @Override
    public ToCWechatCgiBinSignInResponse wechatCgiBinSignIn(ToCWechatCgiBinSignInRequest request) {
        JsCode2SessionResponse jsCode2Session = this.wechatCgiProvider.execJsCode(request.getJsCode());
        if (jsCode2Session == null) {
            throw new BadTenantException(String.format("调用赛演微信获取jsCode异常:code=%s", request.getJsCode()));
        }

        WechatCgiBinSignInCommand command = WechatCgiBinSignInCommand.create(
                jsCode2Session.getOpenId(),
                jsCode2Session.getUnionId(),
                request.getNicky(),
                request.getIpAddress(),
                SystemConstant.SYS_TEMP_SECURITY
        );

        WechatCgiBinSignInCommandResult result = this.identityUserService.dispatch(command);

        String jsonValue = jsonProvider.getJson(WechatCgiBinSubject.create(result.getOpenId(), result.getUnionId(), result.getTenantUserId()));

        String cgiToken = JwtUtil.getMobileToken(jsonValue);
        String token = null;
        /**
         * 表示该C端用户未核验或者未绑定人员
         */
        if (SystemConstant.isEmpty(result.getTenantUserId())) {
            token = null;
        } else {
            token = JwtUtil.getMobileToken(result.getTenantUserId());
        }

        return ToCWechatCgiBinSignInResponse.create(cgiToken, token, result.getNicky());
    }

    @Override
    public void doAssignTocUser(TocUserAssignIdentityRequest request) {
        this.identityUserService.dispatch(TocUserAssignIdentityCommand.create(
                request.getTenantUserId(),
                request.getOpenId(),
                request.getUnionId(),
                request.getName(),
                request.getNumber(),
                request.getPhone()
        ));
    }

    @Override
    public void prepareResetPasswordCheckCode(ResetPasswordRequest request) {
        ResetPasswordCommand command = ResetPasswordCommand.create(
                request.getJwt(), request.getLoginName(),
                null,
                request.getCode()
        );

        this.identityUserService.resetPasswordCheckCode(command);
    }
    @Override
    public void restPassword(ResetPasswordRequest request) {
        ResetPasswordCommand command = ResetPasswordCommand.create(
                request.getJwt(), request.getLoginName(),
                formatFrontPassword(request.isRsa(), request.getNewPassword()),
                request.getCode()
        );

        this.identityUserService.dispatch(command);
    }

    @Override
    public void sendCode(SmsRequest request) {
        smsExecutor.sendCode(request);
    }

    @Override
    public boolean checkCode(SmsRequest request) {
        return smsExecutor.checkCode(request);
    }

    @Override
    public SignInResponse ssoLogin(SsoLoginRequest request) throws Exception {
        SignInCommand command = SignInCommand.create(request.getUserName(),null,false,false, request.getIpAddress());
        SignInCommand.SignInCommandResult result = this.identityUserService.signIn(command);
        return SignInResponse.create(result.getToken(), result.getName(), result.getIdNum(), result.getUpdatePasswordType());
    }

    @Override
    public SignInResponse ssoLoginByIdNum(String userName, String ipAddress) throws Exception {
        SignInCommand command = SignInCommand.create(userName,null,false,false, ipAddress);
        SignInCommand.SignInCommandResult result = this.identityUserService.signIn(command);
        return SignInResponse.create(result.getToken(), result.getName(), result.getIdNum(), result.getUpdatePasswordType());
    }

    @Override
    public SignInResponse signIn(SignInRequest request) {
        SignInCommand command = SignInCommand.create(
                request.getUserName(),
                formatFrontPassword(request.isRsa(), request.getPassword()),
                request.isFromMobile(), request.getIpAddress());

        SignInCommand.SignInCommandResult result = this.identityUserService.signIn(command);

        return SignInResponse.create(result.getToken(), result.getName(), result.getIdNum(), result.getUpdatePasswordType());
    }

    @Override
    public void signUp(SignUpRequest request) {
        Collection<SignUpCommand.CredentialCommandItem> credentialCommandItems = new ArrayList<>();
        if(request.getCredentials()!=null) {
            credentialCommandItems = request.getCredentials().stream().map(ii -> SignUpCommand.CredentialCommandItem.create(ii.getCredentialType(), ii.getNumber()))
                    .collect(Collectors.toList());
        }
        SignUpCommand command = SignUpCommand.create(request.getTenantUserId(), request.getTelephone(),request.getIdNum(), request.getName(),credentialCommandItems);

        this.identityUserService.signUp(command);
    }

    @Override
    public ThirdPartyCommonTokenResponse getAccessToken(ThirdPartyCommonTokenRequest request) {
        return this.wechatCgiProvider.getAccessToken();
    }

    @Override
    public void changedUserName(ChangeUserNameRequest request) {
        TenantUserContext.UserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException("用户信息无效");
        }

        this.identityUserService.dispatch(
                ChangeUserNameCommand.create(
                        userModel.getId(),
                        request.getNewUserName(), request.getPassword())
        );
    }

    private static String formatFrontPassword(boolean isRsa, String password) {
        try {
            if (isRsa) {
                return RSAUtils.decrypt(password);
            }
        } catch (Exception ex) {
            System.err.println(String.format("isRsa=%s;password=%s", isRsa, password));
            ex.printStackTrace();
        }

        return password;
    }
}
