package com.bcxin.rest.web.apis.controllers;

import com.bcxin.Infrastructures.components.CacheV2Provider;
import com.bcxin.Infrastructures.components.RetryProvider;
import com.bcxin.api.interfaces.commons.CommonImportResponse;
import com.bcxin.api.interfaces.tenants.DepartmentRpcProvider;
import com.bcxin.api.interfaces.tenants.requests.DepartmentRequest;
import com.bcxin.api.interfaces.tenants.requests.ImportDepartmentRequest;
import com.bcxin.api.interfaces.tenants.requests.employees.BatchAssignImPermissionRequest;
import com.bcxin.api.interfaces.tenants.requests.employees.BatchMoveDepartmentEmployeeRequest;
import com.bcxin.api.interfaces.tenants.responses.DepartmentGetResponse;
import com.bcxin.api.interfaces.tenants.responses.DepartmentTreeGetResponse;
import com.bcxin.rest.web.apis.ExtractDataComponent;
import com.bcxin.rest.web.apis.requests.BatchDeleteDepartmentRequest;
import com.bcxin.rest.web.apis.requests.BatchImportDataRequest;
import com.bcxin.rest.web.apis.requests.ChangeDepartmentDisplayOrderRequest;
import com.bcxin.rest.web.apis.responses.ApiDepartmentTreeGetResponse;
import com.bcxin.rest.web.apis.utils.CacheUtil;
import com.bcxin.rest.web.apis.utils.FileUtils;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.stream.Collectors;

@Validated
@RestController
@RequestMapping("/tenant/organizations/{organizationId}/departments")
public class DepartmentController extends ControllerAbstract {
    private final DepartmentRpcProvider departmentRpcProvider;
    private final ExtractDataComponent extractDataComponent;
    private final CacheV2Provider cacheV2Provider;
    private final RetryProvider retryProvider;

    public DepartmentController(DepartmentRpcProvider departmentRpcProvider,
                                ExtractDataComponent extractDataComponent,
                                CacheV2Provider cacheV2Provider,RetryProvider retryProvider) {
        this.departmentRpcProvider = departmentRpcProvider;
        this.extractDataComponent = extractDataComponent;
        this.cacheV2Provider = cacheV2Provider;
        this.retryProvider=retryProvider;
    }

    @ApiOperation(value = "获取组织底下的所有部门信息",
            response = ApiDepartmentTreeGetResponse.class,
            responseContainer = "list")
    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            ),
            @ApiImplicitParam(
                    name = "employee_token",
                    value = "组织的token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @GetMapping
    public ResponseEntity<Collection<DepartmentTreeGetResponse>> get(@PathVariable String organizationId) {
        Collection<DepartmentTreeGetResponse> responses = this.cacheV2Provider.get(
                CacheUtil.getDepartmentCacheKey(organizationId, false), () -> {
                    return this.departmentRpcProvider.getDepartments(organizationId);
                }, 2 * 60 * 60);

        return this.ok(responses);
    }

    @ApiOperation(value = "导出部门列表")
    @RequestMapping("/exportExcel")
    public void exportExcel(@PathVariable String organizationId, HttpServletResponse response) throws Exception {
        String filePath = this.departmentRpcProvider.exportExcel(organizationId);
        response.setContentType(MediaType.APPLICATION_OCTET_STREAM_VALUE);
        FileUtils.setAttachmentResponseHeader(response, "部门.xlsx");
        FileUtils.writeBytes(filePath, response.getOutputStream(),retryProvider);
        FileUtils.deleteFile(filePath);
    }

    @ApiOperation(value = "获取部门详细信息", response = DepartmentGetResponse.class)
    @GetMapping("/{id}")
    public ResponseEntity<DepartmentGetResponse> get(@PathVariable String organizationId, @PathVariable String id) {
        DepartmentGetResponse responses =
                this.departmentRpcProvider.getDepartment(organizationId, id);

        return this.ok(responses);
    }


    @ApiOperation(value = "新增部门信息", response = Valid.class)
    @PostMapping
    public ResponseEntity post(@PathVariable String organizationId,
                               @Valid @RequestBody DepartmentRequest request) {
        String departId =
                this.departmentRpcProvider.create(organizationId, request);

        CacheUtil.clearDepartments(cacheV2Provider, organizationId,true);

        return this.ok(departId);
    }

    @ApiOperation(value = "修改部门", response = Valid.class)
    @PutMapping("/{id}")
    public ResponseEntity put(@PathVariable String organizationId, @PathVariable String id, @RequestBody DepartmentRequest request) {
        this.departmentRpcProvider.update(organizationId, id, request);

        CacheUtil.clearDepartments(cacheV2Provider, organizationId,true);
        return this.ok();
    }

    @ApiOperation(value = "删除部门", response = Valid.class)
    @DeleteMapping("/{id}")
    public ResponseEntity delete(@PathVariable String organizationId, @PathVariable String id) {
        this.departmentRpcProvider.delete(organizationId, id);

        CacheUtil.clearDepartments(cacheV2Provider, organizationId,true);
        return this.ok();
    }

    @ApiOperation(value = "批量导入部门", response = Valid.class)
    @PostMapping("/batch-import")
    public ResponseEntity batch_import(
            HttpServletRequest servletRequest,
            @PathVariable String organizationId,
            @Valid @RequestBody BatchImportDataRequest request) {
        Collection<ImportDepartmentRequest> requests = this.extractDataComponent.extract(2, request.getPath(), excelMapValues -> {
            return this.translate2ImportDepartmentRequest(excelMapValues);
        });

        if (requests.isEmpty()) {
            return this.ok("导入数据不能为空");
        }

        CommonImportResponse response = this.departmentRpcProvider.batchImport(organizationId, request.getPath(), requests);
        response.buildDownloadUrl(String.format("%s/download/resources/%s/department", servletRequest.getContextPath(), response.getResourceId()));
        CacheUtil.clearDepartments(cacheV2Provider, organizationId,false);
        return this.ok(response);
    }

    @ApiOperation(value = "转移部门成员", response = Valid.class)
    @PostMapping("/batch-move")
    public ResponseEntity batch_move(@PathVariable String organizationId, @RequestBody BatchMoveDepartmentEmployeeRequest request) {
        this.departmentRpcProvider.batchMove(organizationId, request);

        CacheUtil.clearDepartments(cacheV2Provider, organizationId,true);
        return this.ok();
    }

    @ApiOperation(value = "批量删除部门", response = Valid.class)
    @PostMapping("/batch-delete")
    public ResponseEntity batch_delete(@PathVariable String organizationId, @RequestBody BatchDeleteDepartmentRequest request) {
        this.departmentRpcProvider.batchDelete(organizationId, request.getIds());

        CacheUtil.clearDepartments(cacheV2Provider, organizationId,true);
        return this.ok();
    }

    @ApiOperation(value = "批量配置通讯录信息", response = Valid.class)
    @PostMapping("/batch-assign-im")
    public ResponseEntity batch_assign_im(@PathVariable String organizationId, @RequestBody BatchAssignImPermissionRequest request) {
        this.departmentRpcProvider.batchImPermissions(organizationId, request);

        CacheUtil.clearDepartments(cacheV2Provider, organizationId,true);
        return this.ok();
    }

    @ApiOperation(value = "修改部门顺序信息", response = Valid.class)
    @PostMapping("/{id}/display-order")
    public ResponseEntity changeDisplayOrder(@PathVariable String organizationId,
                                             @PathVariable String id,
                                             @RequestBody ChangeDepartmentDisplayOrderRequest request) {
        this.departmentRpcProvider.update(organizationId, id, request.getDisplayOrder());

        CacheUtil.clearDepartments(cacheV2Provider, organizationId, true);

        return this.ok();
    }

    /**
     * 转换为对象
     * 先跳过前两行
     *
     * @param excelMapValues
     * @return
     */
    private Collection<ImportDepartmentRequest> translate2ImportDepartmentRequest(
            Collection<Map<Integer, String>> excelMapValues) {
        if (CollectionUtils.isEmpty(excelMapValues)) {
            return Collections.emptyList();
        }

        Collection<ImportDepartmentRequest> requests = excelMapValues.parallelStream().map(ii -> {
            String name = StringUtils.trimWhitespace(ii.get(0));
            String parentTreeName = StringUtils.trimWhitespace(ii.get(1));
            String displayOrder = StringUtils.trimWhitespace(ii.get(2));
            String permissionTypeText = StringUtils.trimWhitespace(ii.get(3));
            if (StringUtils.hasLength(name) && StringUtils.hasLength(parentTreeName) && StringUtils.hasLength(displayOrder) &&
                    StringUtils.hasLength(parentTreeName)) {
                return ImportDepartmentRequest.create(name, parentTreeName, displayOrder, permissionTypeText);
            }

            return null;
        }).filter(ii -> ii != null).collect(Collectors.toList());

        return requests;
    }
}
