package com.bcxin.rest.web.apis.controllers;

import cn.hutool.core.util.StrUtil;
import com.bcxin.Infrastructures.Pageable;
import com.bcxin.Infrastructures.components.RetryProvider;
import com.bcxin.Infrastructures.utils.RedisUtil;
import com.bcxin.Infrastructures.utils.UUIDUtil;
import com.bcxin.api.interfaces.commons.CommonImportResponse;
import com.bcxin.api.interfaces.tenants.ContractRpcProvider;
import com.bcxin.api.interfaces.tenants.criterias.ContractCriteria;
import com.bcxin.api.interfaces.tenants.requests.contracts.BatchDeleteContractRequest;
import com.bcxin.api.interfaces.tenants.requests.contracts.BatchImportContractRequest;
import com.bcxin.api.interfaces.tenants.requests.contracts.CreateContractRequest;
import com.bcxin.api.interfaces.tenants.requests.contracts.UpdateContractRequest;
import com.bcxin.api.interfaces.tenants.responses.ContractGetResponse;
import com.bcxin.api.interfaces.tenants.responses.ContractReportResponse;
import com.bcxin.rest.web.apis.ExtractDataComponent;
import com.bcxin.rest.web.apis.requests.BatchImportDataRequest;
import com.bcxin.rest.web.apis.utils.FileUtils;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.stream.Collectors;

@RestController
@RequestMapping("/tenant/organizations/{organizationId}/contracts")
public class ContractController extends ControllerAbstract {

    private final ContractRpcProvider contractRpcProvider;
    private final ExtractDataComponent extractDataComponent;

    private final RedisUtil redisUtil;

    private final   RetryProvider retryProvider;

    public ContractController(ContractRpcProvider contractRpcProvider,
                              ExtractDataComponent extractDataComponent, RedisUtil redisUtil, RetryProvider retryProvider) {
        this.contractRpcProvider = contractRpcProvider;
        this.extractDataComponent = extractDataComponent;
        this.redisUtil = redisUtil;
        this.retryProvider=retryProvider;
    }


    @ApiOperation(value = "劳动合同下载", response = Void.class)
    @RequestMapping  ("/contractDownloadLocal/{id}")
    public ResponseEntity contractDownloadLocal(@PathVariable String organizationId, @PathVariable String id) throws IOException {
        //获取合同文件路径
        String path =contractRpcProvider.getOrganIdAndId(organizationId,id);
        //下载合同文件
        //FileUtils.downloadLocal(path,response);
        return this.ok(path);
    }

    @PostMapping
    public ResponseEntity post(@PathVariable String organizationId, @RequestBody CreateContractRequest request) {
        /*try {
            ServletRequestAttributes attributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
            String sessionId = RequestContextHolder.getRequestAttributes().getSessionId();
            HttpServletRequest request1 = attributes.getRequest();
            String key = sessionId + "-" + request1.getServletPath();
            if(redisUtil.hasKey(key)) {// 如果缓存中有这个url视为重复提交
                throw new RuntimeException("请勿重复提交!");
            } else {
                redisUtil.set(key, UUIDUtil.getShortUuid(), 10);
                this.contractRpcProvider.create(organizationId, request);
            }
        } catch (Throwable e) {
            e.printStackTrace();
        }*/
        this.contractRpcProvider.create(organizationId, request);
        return this.ok();
    }

    @PutMapping("/{id}")
    public ResponseEntity post(@PathVariable String organizationId, @PathVariable String id,
                               @RequestBody UpdateContractRequest request) {
        this.contractRpcProvider.update(organizationId, id, request);

        return this.ok();
    }

    @DeleteMapping("/{id}")
    public ResponseEntity delete(@PathVariable String organizationId, @PathVariable String id) {
        this.contractRpcProvider.delete(organizationId, id);

        return this.ok();
    }

    @PostMapping("/batch-delete")
    public ResponseEntity batchDelete(@PathVariable String organizationId, @RequestBody BatchDeleteContractRequest request) {
        this.contractRpcProvider.batchDelete(organizationId, request);
        return this.ok();
    }

    @ApiOperation(value = "导出合同列表")
    @RequestMapping("/exportExcel")
    public void exportExcel(@PathVariable String organizationId, @RequestBody ContractCriteria criteria, HttpServletResponse response) throws Exception {
        String filePath = this.contractRpcProvider.exportExcel(organizationId,criteria);
        response.setContentType(MediaType.APPLICATION_OCTET_STREAM_VALUE);
        FileUtils.setAttachmentResponseHeader(response, "合同.xlsx");
        FileUtils.writeBytes(filePath, response.getOutputStream(),retryProvider);
        FileUtils.deleteFile(filePath);
    }

    @ApiOperation(value = "获取企业合同列表统计")
    @PostMapping("/searchReport")
    public ResponseEntity searchReport(@PathVariable String organizationId, @Valid @RequestBody ContractCriteria criteria) {
        ContractReportResponse data = this.contractRpcProvider.findReport(organizationId, criteria);
        return this.ok(data);
    }


    @PostMapping("/search")
    public ResponseEntity search(@PathVariable String organizationId, @RequestBody ContractCriteria criteria) {
        Pageable<ContractGetResponse> contractGetResponsePageable =
                this.contractRpcProvider.find(organizationId, criteria);

        return this.ok(contractGetResponsePageable);
    }

    @PostMapping("/batch-import")
    public ResponseEntity batchImport(
            HttpServletRequest servletRequest,
            @PathVariable String organizationId,
                                      @RequestBody BatchImportDataRequest request ) {
        Collection<BatchImportContractRequest> batchImportContractRequests =
                extractDataComponent.extract(2, request.getPath(), excelMapValues -> {
                    return translate2BatchImportContractRequests(excelMapValues);
                });

        CommonImportResponse response =
                this.contractRpcProvider.batchImport(organizationId, request.getPath(), batchImportContractRequests);

        response.buildDownloadUrl(
                String.format("%s/download/resources/%s/contract", servletRequest.getContextPath(), response.getResourceId())
        );

        return this.ok(response);
    }

    /**
     * 转换为对象
     * 先跳过前两行
     *
     * @param excelMapValues
     * @return
     */
    private Collection<BatchImportContractRequest> translate2BatchImportContractRequests(
            Collection<Map<Integer, String>> excelMapValues) {
        if (CollectionUtils.isEmpty(excelMapValues)) {
            return Collections.emptyList();
        }

        Collection<BatchImportContractRequest> requests = excelMapValues.parallelStream().map(ii -> {
            String name = StringUtils.trimWhitespace(ii.get(0));
            String aName = StringUtils.trimWhitespace(ii.get(1));
            String bName = StringUtils.trimWhitespace(ii.get(2));
            String idNum = StringUtils.trimWhitespace(ii.get(3));
            String beginDate = StringUtils.trimWhitespace(ii.get(4));
            String endDate = StringUtils.trimWhitespace(ii.get(5));

            if(StrUtil.isEmpty(name)){
                name = bName + "的劳动合同";
            }

            if (StringUtils.hasLength(name) && StringUtils.hasLength(idNum)) {

                return BatchImportContractRequest.create(name, aName, bName, idNum, beginDate, endDate);
            }

            return null;
        }).filter(ii -> ii != null).collect(Collectors.toList());

        return requests;
    }
}
