package com.bcxin.Infrastructures.utils;

import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.Date;

public class RetirementAgeUtil {

    /**
     * 计算退休日期
     *
     * @param birthDate             出生日期
     * @param originalRetirementAge 原退休年龄
     * @return {@link LocalDate }
     */
    public static LocalDate calculateRetirementDate(LocalDate birthDate, int originalRetirementAge) {
        // 基本设置
        LocalDate policyStartDate = LocalDate.of(2025, 1, 1);

        // 计算原退休日期
        LocalDate originalRetirementDate = birthDate.plusYears(originalRetirementAge);

        // 计算政策开始日期到原退休日期的月数差异
        long monthsBetween = ChronoUnit.MONTHS.between(policyStartDate, originalRetirementDate)+1L;
        long daysBetween = ChronoUnit.DAYS.between(policyStartDate, originalRetirementDate);

        // 如果月数差异为0或负值，直接返回原退休日期
        if (daysBetween < 0) {
            return originalRetirementDate;
        }

        // 计算调整后月份
        int monthsToAdd = 0;

        switch (originalRetirementAge) {
            case 60:
            case 55:
                monthsToAdd = (int) Math.ceil((double) monthsBetween / 4);
                if (monthsToAdd > 36) {
                    monthsToAdd = 36;
                }
                break;
            case 50:
                // 原法定退休年龄为50岁的女职工：每2个月延迟1个月，逐步延迟至55岁
                monthsToAdd = (int) Math.ceil((double) monthsBetween / 2);
                if (monthsToAdd > 60) {
                    monthsToAdd = 60;
                }
                break;
            default:
                break;
        }

        // 计算最终的退休日期
        return originalRetirementDate.plusMonths(monthsToAdd);
    }

    /**
     * description：是否超龄
     * author：linchunpeng
     * date：2025/5/15
     */
    public static boolean isExceedRetirementAge(Date birthDate, int originalRetirementAge) {
        SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
        String formattedDate = formatter.format(birthDate);
        LocalDate localDate = LocalDate.parse(formattedDate);
        LocalDate retirementDate = calculateRetirementDate(localDate, originalRetirementAge);
        return retirementDate.isBefore(LocalDate.now());
    }

}
