package com.bcxin.Infrastructures.enums;

import cn.hutool.core.util.IdcardUtil;
import com.bcxin.Infrastructures.exceptions.ArgumentTenantException;
import com.bcxin.Infrastructures.utils.AuthUtil;
import com.bcxin.Infrastructures.utils.RetirementAgeUtil;
import com.bcxin.Infrastructures.validations.CredentialValidator;
import com.bcxin.Infrastructures.validations.DateValidator;
import org.springframework.util.StringUtils;

import java.util.Date;

public enum OccupationType {
    /**
     * 一般职员
     */
    Normal {
        @Override
        public String getTypeName() {
            return "一般职员";
        }
        @Override
        public void validate(CredentialType credentialType, String number, String areaCode, String institutionalCode) {
            super.validate(credentialType, number, areaCode, institutionalCode);
            if (institutionalCode == null || !institutionalCode.startsWith("02")) {
                if (credentialType == CredentialType.IdCard) {
                    Date d=new Date();
                    Date birthDate =IdcardUtil.getBirthDate(number);
                    if (DateValidator.dateAddYear(birthDate,18).after(d)){
                        throw new ArgumentTenantException("一般职员, 年龄必须大于等于18岁");
                    }
                }
            }
        }
    },
    /**
     * 保安员
     */
    SecurityGuard {
        @Override
        public String getTypeName() {
            return "保安员";
        }

        @Override
        public void validate(CredentialType credentialType, String number, String areaCode, String institutionalCode) {
            super.validate(credentialType, number, areaCode, institutionalCode);
            Date d = new Date();
            d.setHours(0);
            d.setMinutes(0);
            d.setSeconds(0);
            Date birthDate = IdcardUtil.getBirthDate(number);
            if (institutionalCode == null || !institutionalCode.startsWith("02")) {
                if (DateValidator.dateAddYear(birthDate, 18).after(d)) {
                    throw new ArgumentTenantException("入职保安员, 年龄必须大于等于18岁");
                }
            }
            if (institutionalCode == null || !institutionalCode.startsWith("02")) {
                //不是大活行业，才需要限制年龄
                if (AuthUtil.isBeijingCompany(areaCode)) {
                    if (IdcardUtil.getGenderByIdCard(number) == 0) {
                        if (RetirementAgeUtil.isExceedRetirementAge(birthDate, 50)) {
                            throw new ArgumentTenantException("根据最新人社部规定，该人员已超龄");
                        }
                    } else if (RetirementAgeUtil.isExceedRetirementAge(birthDate, 60)) {
                        throw new ArgumentTenantException("根据最新人社部规定，该人员已超龄");
                    }
                } else if (AuthUtil.isHunanCompany(areaCode)) {
                    if (IdcardUtil.getGenderByIdCard(number) == 0) {
                        if (DateValidator.dateAddYear(birthDate, 60).before(d)) {
                            throw new ArgumentTenantException("入职女保安员, 年龄必须小于等于60岁");
                        }
                    } else if (DateValidator.dateAddYear(birthDate, 60).before(d)) {
                        throw new ArgumentTenantException("入职男保安员, 年龄必须小于等于60岁");
                    }
                } else if (AuthUtil.isNingxiaCompany(areaCode)) {
                    if (IdcardUtil.getGenderByIdCard(number) == 0) {
                        if (DateValidator.dateAddYear(birthDate, 55).before(d)) {
                            throw new ArgumentTenantException("入职女保安员, 年龄必须小于等于55岁");
                        }
                    } else if (DateValidator.dateAddYear(birthDate, 60).before(d)) {
                        throw new ArgumentTenantException("入职男保安员, 年龄必须小于等于60岁");
                    }
                } else if (AuthUtil.isShanxiCompany(areaCode)) {
                    if (IdcardUtil.getGenderByIdCard(number) == 0) {
                        if (DateValidator.dateAddYear(birthDate, 58).before(d)) {
                            throw new ArgumentTenantException("入职女保安员, 年龄必须小于等于58岁");
                        }
                    } else if (DateValidator.dateAddYear(birthDate, 63).before(d)) {
                        throw new ArgumentTenantException("入职男保安员, 年龄必须小于等于63岁");
                    }
                }
            }
        }
    },
    /**
     * 警员
     */
    Police {
        @Override
        public String getTypeName() {
            return "警员";
        }
    };

    public abstract String getTypeName();

    public void validate(CredentialType credentialType, String number, String areaCode, String institutionalCode) {
        /**
         * 默认情况下, 只对身份证的有效性进行验证
         */
        if (credentialType == CredentialType.IdCard) {
            if (!StringUtils.hasLength(number)) {
                throw new ArgumentTenantException("身份证不能为空");
            }

            if (!CredentialValidator.validate(CredentialType.IdCard, number)) {
                throw new ArgumentTenantException("身份证格式不正确");
            }
        }
    }
}
