package com.bcxin.tenant.bcx.domains.repositories;

import com.bcxin.tenant.bcx.domains.components.CacheProvider;
import com.bcxin.tenant.bcx.domains.entities.MetaPageEntity;
import com.bcxin.tenant.bcx.domains.entities.MetaPageTableMappingEntity;
import com.bcxin.tenant.bcx.domains.mappers.MetaPageTableMappingMapper;
import com.bcxin.tenant.bcx.domains.utils.BusinessConstants;
import com.bcxin.tenant.bcx.infrastructures.components.JsonProvider;
import com.bcxin.tenant.bcx.infrastructures.enums.DataStatus;
import com.bcxin.tenant.bcx.infrastructures.enums.DynamicMappingType;
import com.bcxin.tenant.bcx.infrastructures.enums.PageBusinessType;
import com.bcxin.tenant.bcx.infrastructures.exceptions.ArgumentTenantException;
import com.bcxin.tenant.bcx.infrastructures.exceptions.NoFoundTenantException;
import com.bcxin.tenant.bcx.infrastructures.exceptions.NotSupportTenantException;
import com.bcxin.tenant.bcx.domains.snapshots.DynamicPageTableMappingDataSnapshot;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

@Repository
public class MetaPageTableMappingRepositoryImpl extends RepositoryAbstract<MetaPageTableMappingEntity>
        implements MetaPageTableMappingRepository {

    private final CacheProvider cacheProvider;
    private final JsonProvider jsonProvider;
    private final MetaPageRepository metaPageRepository;

    protected MetaPageTableMappingRepositoryImpl(MetaPageTableMappingMapper mapper,
                                                 CacheProvider cacheProvider,
                                                 JsonProvider jsonProvider,
                                                 MetaPageRepository metaPageRepository) {
        super(mapper);
        this.cacheProvider = cacheProvider;
        this.jsonProvider = jsonProvider;
        this.metaPageRepository = metaPageRepository;
    }

    @Override
    public DynamicPageTableMappingDataSnapshot getSnapshot(String pageId) {
        if (pageId == null) {
            return null;
        }

        MetaPageEntity pageEntity = metaPageRepository.getById(pageId);
        if (pageEntity == null) {
            throw new NoFoundTenantException(String.format("页面不存在(%s)", pageId));
        }

        String mappingId = pageEntity.getTableName();
        if (!StringUtils.hasLength(mappingId)) {
            if (pageEntity.getBusinessType() == PageBusinessType.ExamEnrollmentCriteria) {
                mappingId = BusinessConstants.SYS_DYNAMIC_EXAM_ENROLLMENT_MAPPING_ID;
            } else {
                mappingId = pageId;
            }
        }

        return getSnapshotByMappingId(mappingId);
    }

    @Override
    public DynamicPageTableMappingDataSnapshot getSnapshotByMappingId(String mappingId) {
        DynamicPageTableMappingDataSnapshot snapshot =
                this.cacheProvider.get(getPageTableMappingCacheKey(mappingId), () -> {
                    /**
                     * 根据MappingId获取内容
                     */
                    MetaPageTableMappingEntity entity = this.getById(mappingId);
                    if (entity == null) {
                        return DynamicPageTableMappingDataSnapshot.empty();
                    }

                    if(entity.getStatus()!= DataStatus.Normal) {
                        return null;
                    }

                    DynamicPageTableMappingDataSnapshot sn =
                            this.jsonProvider.toObject(DynamicPageTableMappingDataSnapshot.class, entity.getMappingJson());

                    return sn;
                }, DynamicPageTableMappingDataSnapshot.class, 5 * 1);

        if (!DynamicPageTableMappingDataSnapshot.isEmpty(snapshot)) {
            if (snapshot.getMappingType() == DynamicMappingType.Sql) {
                /*
                if (snapshot.getDsl() == null || snapshot.getDsl() == null ||
                        !StringUtils.hasLength(snapshot.getDsl().getInsertDsl()) ||
                        !StringUtils.hasLength(snapshot.getDsl().getUpdateDsl()) ||
                        !StringUtils.hasLength(snapshot.getDsl().getDeleteDsl())) {
                    throw new ArgumentTenantException(String.format("参数异常(%s), 请联系管理员", mappingId));
                }
                 */
            } else {
                throw new NotSupportTenantException(String.format("不支持此配置(%s)", mappingId));
            }
        }

        return snapshot;
    }

    @Override
    public void refreshCache(String pageId) {
        this.cacheProvider.delete(getPageTableMappingCacheKey(pageId));
    }

    private static String getPageTableMappingCacheKey(String pageId) {
        return String.format("paas:bcx:snapshot:pageTableMapping:%s:v2", pageId);
    }
}
