package com.bcxin.tenant.bcx.infrastructures;

import com.bcxin.tenant.bcx.infrastructures.constants.BusinessConstants;
import com.bcxin.tenant.bcx.infrastructures.enums.MetaCreatorReferenceType;
import com.bcxin.tenant.bcx.infrastructures.exceptions.ArgumentTenantException;
import com.bcxin.tenant.bcx.infrastructures.utils.StringUtil;
import lombok.AccessLevel;
import lombok.Getter;

import java.io.Serializable;
import java.sql.Timestamp;
import java.time.Instant;
import java.util.*;
import java.util.function.Supplier;
import java.util.stream.Collectors;

public class TenantEmployeeContext implements ContextAware<TenantEmployeeContext.TenantUserModel> {
    public static final String CURRENT_TENANT_USER_INFO = "API.CURRENT_TENANT_USER_INFO";
    private static final ThreadLocal<TenantUserModel> _userContextThread = new ThreadLocal<>();
    private Supplier<Collection<TenantDeviceDeskServiceScopeModel>> deviceDeskServiceScopesSupplier;

    private Supplier<TenantSecurityEmployeeModel> tenantSecurityEmployeeSupplier;


    @Override
    public TenantUserModel get() {
        return _userContextThread.get();
    }

    @Override
    public void init(TenantUserModel data) {
        _userContextThread.set(data);
    }

    public TenantUserModel init(
            UserDetailResponse userDetailResponse,
            PermissionModel permissionModel
    ) {
        TenantUserModel userModel =
                TenantUserModel.create(
                        userDetailResponse.getId(),
                        userDetailResponse.getEmployeeId(),
                        userDetailResponse.getName(),
                        userDetailResponse.getIdCardNo(),
                        userDetailResponse.getOrganizationId(),
                        userDetailResponse.getCLevel(),
                        userDetailResponse.getImNum(),
                        userDetailResponse.getSupervise(),
                        permissionModel
                );

        userModel.assignOrganizationInfo(TenantEmployeeContext.OrganizationInfoModel.create(
                userDetailResponse.getOrgName(),
                userDetailResponse.getOrgLatitude(),
                userDetailResponse.getOrgLongitude(),
                userDetailResponse.getInstitutional(),
                userDetailResponse.getSuperviseRegionCode()
        ));

        _userContextThread.set(userModel);

        return userModel;
    }

    public void reset() {
        _userContextThread.set(null);
    }

    @Getter
    public static class TenantUserModel implements Serializable {
        private static final long serialVersionUID = 8880000001L;
        private String id;
        private final String employeeId;
        private final String name;
        private final String idCardNo;
        private final String organizationId;
        private final int cLevel;

        private final PermissionModel permission;

        /**
         * 智能人事的im_identity
         */
        private final String imNum;

        private final boolean supervise;

        private Collection<TenantDeviceDeskServiceScopeModel> deviceDeskServiceScopes;

        private TenantSecurityEmployeeModel tenantSecurityEmployee;

        @Getter(AccessLevel.PRIVATE)
        private OrganizationInfoModel organizationInfo;

        public MetaCreatorReferenceType getUserType(){
            return MetaCreatorReferenceType.Platform;
        }

        public Map<String,Object> fillAndReturnMap(Map<String,Object> map)
        {
            TenantUserModel userModel = this;
            Map<String, Object> userInfo = new HashMap<>();
            userInfo.put("id", userModel.getId());
            userInfo.put("name", userModel.getName());
            userInfo.put("organizationId", userModel.getOrganizationId());
            userInfo.put("domainId", userModel.getOrganizationId());
            userInfo.put("cLevel", userModel.getCLevel());
            userInfo.put("imNum", userModel.getImNum());
            userInfo.put("supervise", userModel.isSupervise());
            userInfo.put("employeeId", userModel.getEmployeeId());
            userInfo.put("idCardNo", userModel.getIdCardNo());

            map.put("current", userInfo);
            map.put("current_idCardNo", userModel.getIdCardNo());
            map.put("current_id", userModel.getId());
            map.put("current_organizationId", userModel.getOrganizationId());

            return map;
        }

        public Double getOrgLatitude() {
            if (this.organizationInfo == null) {
                return null;
            }

            return this.organizationInfo.getLatitude();
        }

        public Double getOrgLongitude() {
            if (this.organizationInfo == null) {
                return null;
            }

            return this.organizationInfo.getLongitude();
        }

        public String getOrgInstitutional() {
            if (this.organizationInfo == null) {
                return null;
            }

            return this.organizationInfo.getInstitutional();
        }

        public String getSuperviseRegionCode() {
            if (this.organizationInfo == null) {
                return null;
            }

            return this.organizationInfo.getSuperviseRegionCode();
        }

        private String trafficTag;
        public String getTrafficTag() {
            if (trafficTag == null) {
                try {
                    trafficTag = this.getSuperviseRegionCode().substring(0, 3);
                } catch (Exception ex) {
                    //todo: 根据区域获取流量标签信息
                }
            }

            return trafficTag;
        }

        public String getOrgName() {
            if (this.organizationInfo == null) {
                return null;
            }

            return this.organizationInfo.getName();
        }

        public boolean isDomainAdmin() {
            PermissionModel permissionModel = this.getPermission();
            if (permissionModel == null) {
                return false;
            }

            return this.getPermission().isDomainAdmin();
        }

        /**
         * 获取非监管的数据权限范围
         * todo: 后期如果企业的非常多, 比如: 大于几百家; 那么考虑优化中需要提交大脚本到服务器进行查询的问题
         * @return
         */
        public String[] getNoSuperviseScopePermissions() {
            Collection<String> allScopes = new HashSet<>();
            if (this.permission!=null &&  this.permission.getSubScopeCompanyIds() != null) {
                allScopes.addAll(this.permission.getSubScopeCompanyIds());
            }

            allScopes.add(this.getOrganizationId());

            return allScopes.toArray(ix -> new String[ix]);
        }

        public void assignDeviceDeskServiceScopes(Collection<TenantDeviceDeskServiceScopeModel> deviceDeskServiceScopes) {
            this.deviceDeskServiceScopes = deviceDeskServiceScopes;
        }

        public void assignTenantSecurityEmployee(TenantSecurityEmployeeModel tenantSecurityEmployee) {
            this.tenantSecurityEmployee = tenantSecurityEmployee;
        }

        public void assignOrganizationInfo(OrganizationInfoModel organizationInfo) {
            this.organizationInfo = organizationInfo;
        }

        /**
         * 有效的调度台账号
         *
         * @return
         */
        public String[] getAssignedSuperviseDepartIds() {
            if (this.getDeviceDeskServiceScopes() == null) {
                return new String[0];
            }

            Collection<String> superviseDepartIds =
                    this.getValidDeskServiceScopes().stream()
                            .map(ix -> ix.getAssignedSuperviseDepartId())
                            .filter(ix -> !StringUtil.isEmpty(ix) && !ix.equalsIgnoreCase(BusinessConstants.DefaultEmptyValue))
                            .collect(Collectors.toList());

            return superviseDepartIds.toArray(new String[superviseDepartIds.size()]);
        }

        private Collection<TenantDeviceDeskServiceScopeModel> getValidDeskServiceScopes() {
            if (this.getDeviceDeskServiceScopes() == null || this.getDeviceDeskServiceScopes().isEmpty()) {
                return Collections.EMPTY_LIST;
            }

            Timestamp now = Timestamp.from(Instant.now());
            return this.getDeviceDeskServiceScopes()
                    .stream().filter(ix -> {
                        Timestamp beginDate = ix.getBeginDate();
                        if (beginDate == null) {
                            beginDate = Timestamp.from(Instant.MIN);
                        }

                        boolean flag = (beginDate.before(now) && ix.getEndDate().after(now)) || (ix.getEndDate() == null);

                        return flag;
                    }).collect(Collectors.toList());
        }

        public Collection<String> getRelativeSameSelectedDepartIds(Collection<String> departIds) {
            if (departIds == null || departIds.isEmpty()) {
                return departIds;
            }

            Collection<TenantDeviceDeskServiceScopeModel> scopeModels = this.getValidDeskServiceScopes();
            if (scopeModels == null || scopeModels.isEmpty()) {
                return departIds;
            }

            /**
             * RdDeviceDeskServiceScopeValueType copy(String superviseDepartId)
             * 比较逻辑来自于上面的方法, 只有同名的才会出现一致的情况
             */
            Set<String> unionMatchedDepartIds = scopeModels.stream()
                    .filter(ii -> departIds.contains(ii.getId()))
                    .map(ii -> ii.getAssignedSuperviseDepartId())
                    .collect(Collectors.toSet());

            Set<String> mergedDepartIds = new HashSet<>();
            mergedDepartIds.addAll(departIds);
            mergedDepartIds.addAll(unionMatchedDepartIds);

            return mergedDepartIds;
        }

        public boolean isSuperviseDepartRole() {
            return this.isSupervise();
        }

        public String getSecurityStationId() {
            TenantSecurityEmployeeModel securityEmployee = this.getTenantSecurityEmployee();
            if (securityEmployee == null) {
                return null;
            }

            return securityEmployee.getSecurityStationId();
        }

        public String getSuperviseDepartId() {
            TenantSecurityEmployeeModel securityEmployee = this.getTenantSecurityEmployee();
            if (securityEmployee == null) {
                return null;
            }

            return securityEmployee.getSuperviseDepartId();
        }

        public String getSuperviseDepartName() {
            TenantSecurityEmployeeModel securityEmployee = this.getTenantSecurityEmployee();
            if (securityEmployee == null) {
                return null;
            }

            return securityEmployee.getSuperviseDepartName();
        }

        public String getTenantUserIdCard() {
            TenantSecurityEmployeeModel securityEmployee = this.getTenantSecurityEmployee();
            if (securityEmployee == null) {
                return null;
            }

            return securityEmployee.getTenantUserIdCard();
        }

        private TenantUserModel(String id, String employeeId, String name,
                                String idCardNo,
                                String organizationId,
                                int cLevel,
                                String imNum,
                                Boolean supervise, PermissionModel permission) {
            this.id = id;
            this.employeeId = employeeId;
            this.name = name;
            this.idCardNo = idCardNo;
            this.organizationId = organizationId;
            this.cLevel = cLevel;
            this.permission = permission;
            this.imNum = imNum;
            this.supervise = supervise == null ? false : supervise.booleanValue();
        }


        public static TenantUserModel create(String id,
                                             String employeeId, String name,
                                             String idCardNo,
                                             String organizationId,
                                             int cLevel,
                                             String imNum,
                                             Boolean supervise, PermissionModel permission) {
            return new TenantUserModel(
                    id, employeeId, name,idCardNo, organizationId, cLevel,
                    imNum, supervise, permission);
        }
    }

    @Getter
    public static class TenantDeviceDeskServiceScopeModel implements Serializable {
        private static final long serialVersionUID = 8880000002L;
        /**
         * 设备Id
         */
        private final String id;

        private final String name;

        /**
         * 腾讯的资源号
         */
        private final String number;
        /**
         * 有效期开始时间
         */
        private final Timestamp beginDate;
        /**
         * 有效期结束时间
         */
        private final Timestamp endDate;
        /**
         * 被绑定的人
         */
        private final String boundEmployeeId;

        /**
         * 被授权的监管归属的部门Id
         */
        private final String assignedSuperviseDepartId;

        /**
         * Ip白名单
         */
        private final String ipAddress;

        private final String organizationId;

        private final byte ir;

        public boolean isActive() {
            if (this.getBeginDate() == null || this.getEndDate() == null) {
                return false;
            }

            Date currentDate = Calendar.getInstance().getTime();

            /**
             * todo 暂时永不过期
             */
            //return true;
            return currentDate.after(this.getBeginDate()) && currentDate.before(this.getEndDate());
        }

        public TenantDeviceDeskServiceScopeModel(String id,
                                                 String name,
                                                 String number,
                                                 Timestamp beginDate, Timestamp endDate,
                                                 String boundEmployeeId,
                                                 String assignedSuperviseDepartId,
                                                 String ipAddress,
                                                 String organizationId,
                                                 byte ir) {
            this.id = id;
            this.name = name;
            this.number = number;
            this.beginDate = beginDate;
            this.endDate = endDate;
            this.boundEmployeeId = boundEmployeeId;
            this.assignedSuperviseDepartId = assignedSuperviseDepartId;
            this.ipAddress = ipAddress;
            this.organizationId = organizationId;
            this.ir = ir;
        }

        public static TenantDeviceDeskServiceScopeModel create(String id,
                                                               String name,
                                                               String number,
                                                               Timestamp beginDate, Timestamp endDate,
                                                               String boundEmployeeId,
                                                               String assignedSuperviseDepartId,
                                                               String ipAddress,
                                                               String organizationId,
                                                               byte ir) {
            return new TenantDeviceDeskServiceScopeModel(id,
                    name,
                    number, beginDate, endDate,
                    boundEmployeeId, assignedSuperviseDepartId, ipAddress, organizationId, ir);
        }

        @Override
        public String toString() {
            return String.format("调度台:%s;有效期:%s~%s", this.getNumber(), this.getBeginDate(), this.getEndDate());
        }
    }

    /**
     * 仅APP的时候才会取这个值; 因此, 需要特别注意
     */
    @Getter
    public static class TenantSecurityEmployeeModel implements Serializable {
        private static final long serialVersionUID = 8880000003L;
        /**
         * 最新的驻勤信息
         */
        private final String securityStationId;
        /**
         * 最新的驻勤名称
         */
        private final String securityStationName;
        /**
         * 最新的监管归属信息
         */
        private final String superviseDepartId;

        /**
         * 最新的监管归属名称
         */
        private final String superviseDepartName;
        /**
         * 最新的腾讯资源信息
         */
        private final String tencentUserId;

        /**
         * 用户姓名
         */
        private final String tenantUserName;

        /**
         * 用户身份证
         */
        private final String tenantUserIdCard;
        private final Set<String> resourceTypes;

        private final Timestamp dataCreatedTime;

        public TenantSecurityEmployeeModel(String securityStationId, String securityStationName,
                                           String superviseDepartId, String tencentUserId,
                                           String tenantUserName, String tenantUserIdCard,
                                           String superviseDepartName,
                                           Set<String> resourceTypes,
                                           Timestamp dataCreatedTime) {
            this.securityStationId = securityStationId;
            this.securityStationName = securityStationName;
            this.superviseDepartId = superviseDepartId;
            this.tencentUserId = tencentUserId;
            this.tenantUserName = tenantUserName;
            this.tenantUserIdCard = tenantUserIdCard;
            this.superviseDepartName = superviseDepartName;
            this.resourceTypes = resourceTypes;
            this.dataCreatedTime = dataCreatedTime;
        }

        public static TenantSecurityEmployeeModel create(String securityStationId, String securityStationName,
                                                         String superviseDepartId, String tencentUserId,
                                                         String tenantUserName, String tenantUserIdCard,
                                                         String superviseDepartName,
                                                         Set<String> resourceTypes,
                                                         Timestamp dataCreatedTime) {
            return new TenantSecurityEmployeeModel(
                    securityStationId, securityStationName, superviseDepartId, tencentUserId,
                    tenantUserName, tenantUserIdCard, superviseDepartName,resourceTypes,
                    dataCreatedTime
            );
        }
    }

    @Getter
    public static class OrganizationInfoModel implements Serializable {
        private static final long serialVersionUID = 8880000004L;
        private Double latitude;

        private Double longitude;

        private String institutional;

        private String superviseRegionCode;

        private String name;

        public static OrganizationInfoModel create(
                String name,
                Double latitude, Double longitude, String institutional,
                String superviseRegionCode) {
            OrganizationInfoModel organization = new OrganizationInfoModel();
            organization.institutional = institutional;
            organization.latitude = latitude;
            organization.longitude = longitude;
            organization.name = name;
            organization.superviseRegionCode = superviseRegionCode;

            return organization;
        }
    }

    @Getter
    public static class PermissionModel implements Serializable {
        private static final long serialVersionUID = 8880000005L;
        private final boolean domainAdmin;
        /**
         * 限制查看的驻勤点信息
         */
        private final Set<String> responsibleOfStationIds;

        /**
         * todo: 20241214: 授权承办单位所管理的岗点列表（授权; 承办单位可以看到关联的所有岗点，但是这边需要过滤出允许的子集）
         */
        private final List<String> selectedStationIds;

        private final Set<String> subScopeCompanyIds;

        private final Timestamp createdTime;

        private final String trace;

        public PermissionModel(boolean domainAdmin, Set<String> responsibleOfStationIds,List<String> selectedStationIds,Collection<String> subScopeCompanyIds,
                               String trace) {
            this.domainAdmin = domainAdmin;
            if(responsibleOfStationIds!=null) {
                responsibleOfStationIds = responsibleOfStationIds.stream()
                        .filter(ii -> !BusinessConstants.DefaultEmptyValue.equalsIgnoreCase(ii))
                        .collect(Collectors.toSet());
            }

            this.responsibleOfStationIds = responsibleOfStationIds;
            if (subScopeCompanyIds == null) {
                this.subScopeCompanyIds = new HashSet<>();
            } else {
                this.subScopeCompanyIds = subScopeCompanyIds.stream().collect(Collectors.toSet());
            }

            if (selectedStationIds == null) {
                this.selectedStationIds = new ArrayList<>();
            } else {
                /**
                 * 过滤非项目信息
                 */
                this.selectedStationIds = selectedStationIds.stream().filter(ii -> !BusinessConstants.isRelativeProjectIdStuffId(ii))
                        .filter(ii -> !ii.equalsIgnoreCase(BusinessConstants.DefaultEmptyValue)).collect(Collectors.toList());
            }

            this.trace = trace;
            this.createdTime = Timestamp.from(Instant.now());
        }

        public static PermissionModel create(boolean domainAdmin, Set<String> responsibleOfStationIds,
                                             List<String> selectedStationIds,
                                             Collection<String> subScopeCompanyIds,
                                             String trace) {
            return new PermissionModel(domainAdmin,responsibleOfStationIds,selectedStationIds,subScopeCompanyIds,trace);
        }
    }


}
