package com.bcxin.tenant.bcx.domains.services.impls;

import com.bcxin.tenant.bcx.domains.entities.MetaProjectEntity;
import com.bcxin.tenant.bcx.domains.repositories.MetaProjectRepository;
import com.bcxin.tenant.bcx.domains.repositories.MetaModuleRepository;
import com.bcxin.tenant.bcx.domains.repositories.criterias.MetaModuleSearchCriteria;
import com.bcxin.tenant.bcx.domains.services.MetaProjectService;
import com.bcxin.tenant.bcx.domains.services.commands.DeleteDataCommand;
import com.bcxin.tenant.bcx.domains.services.commands.projects.BatchLogicalDeleteProjectCommand;
import com.bcxin.tenant.bcx.domains.services.commands.projects.CreateMetaProjectCommand;
import com.bcxin.tenant.bcx.domains.services.commands.projects.LogicalDeleteProjectCommand;
import com.bcxin.tenant.bcx.domains.services.commands.projects.UpdateMetaProjectCommand;
import com.bcxin.tenant.bcx.infrastructures.EntityCollection;
import com.bcxin.tenant.bcx.infrastructures.TenantContext;
import com.bcxin.tenant.bcx.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.bcx.infrastructures.UnitWork;
import com.bcxin.tenant.bcx.infrastructures.components.IdWorker;
import com.bcxin.tenant.bcx.infrastructures.exceptions.NoFoundTenantException;
import com.bcxin.tenant.bcx.infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.tenant.bcx.infrastructures.utils.StringUtil;
import org.springframework.stereotype.Service;

@Service
public class MetaProjectServiceImpl implements MetaProjectService {

    private final MetaProjectRepository metaProjectRepository;
    private final MetaModuleRepository metaModuleRepository;
    private final UnitWork unitWork;

    private final IdWorker idWorker;

    public MetaProjectServiceImpl(MetaProjectRepository metaProjectRepository, 
                                  MetaModuleRepository metaModuleRepository,
                                  UnitWork unitWork, 
                                  IdWorker idWorker) {
        this.metaProjectRepository = metaProjectRepository;
        this.metaModuleRepository = metaModuleRepository;
        this.unitWork = unitWork;
        this.idWorker = idWorker;
    }

    @Override
    public void dispatch(CreateMetaProjectCommand command) {
        command.validate();

        if(StringUtil.isEmpty(command.getName())){
            throw new IllegalArgumentException("项目名称不能为空");
        }

        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if(userModel==null) {
            throw new UnAuthorizedTenantException();
        }

        this.unitWork.executeTran(() -> {

            String id = command.getId();

            if (StringUtil.isEmpty(command.getId())){
                // 生成ID
                id = String.valueOf(this.idWorker.getNextId());
            }

            // 创建项目实体 - 字段值由AOP切面自动填充
            MetaProjectEntity entity = MetaProjectEntity.create(
                id, 
                command.getName(), 
                command.getNote(), 
                command.isEnabled(), 
                command.getLogo(), 
                command.getSlug(),
                userModel.getUserType(),
                userModel.getOrganizationId(),
                userModel.getEmployeeId()
            );
            // 保存到数据库
            this.metaProjectRepository.insert(entity);
        });
    }

    @Override
    public void dispatch(UpdateMetaProjectCommand command) {
        command.validate();

        this.unitWork.executeTran(() -> {
            // 根据ID查询现有实体（包括已删除的）
            MetaProjectEntity entity = getByIdIncludeDeleted(command.getId());
            if (entity == null) {
                throw new NoFoundTenantException();
            }

            // 检查是否已删除
            if (entity.isDeleted()) {
                throw new NoFoundTenantException();
            }

            // 更新字段 - lastUpdaterId由AOP切面自动填充
            TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
            entity.change(
                    command.getName(),
                    command.getNote(),
                    command.isEnabled(),
                    command.getLogo(),
                    command.getSlug(),
                    userModel
            );

            // 保存更新
            this.metaProjectRepository.update(entity);
        });
    }

    @Override
    public void dispatch(DeleteDataCommand command) {
        command.validate();

        this.unitWork.executeTran(() -> {
            // 根据ID查询现有实体（包括已删除的）
            MetaProjectEntity entity = getByIdIncludeDeleted(command.getId());
            if (entity == null) {
                throw new NoFoundTenantException();
            }

            // 检查是否已删除
            if (entity.isDeleted()) {
                throw new NoFoundTenantException();
            }

            // 检查项目下是否有模块
            MetaModuleSearchCriteria criteria = MetaModuleSearchCriteria.create(
                null, null, command.getId(), null, 1, 1
            );
            EntityCollection<?> modules = this.metaModuleRepository.search(criteria);
            
            if (modules.getTotalCount() > 0) {
                throw new IllegalArgumentException("无法删除项目，该项目下还有 " + modules.getTotalCount() + " 个模块，请先删除所有模块");
            }

            // 删除项目
            this.metaProjectRepository.delete(command.getId());
        });
    }

    @Override
    public void dispatch(LogicalDeleteProjectCommand command) {
        command.validate();

        this.unitWork.executeTran(() -> {
            // 根据ID查询现有实体（包括已删除的）
            MetaProjectEntity entity = getByIdIncludeDeleted(command.getId());
            if (entity == null) {
                throw new NoFoundTenantException();
            }

            // 检查项目是否已删除
            if (entity.isDeleted()) {
                throw new NoFoundTenantException();
            }

            // 标记为已删除
            TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
            entity.changeMarkAsDeleted(userModel);

            // 保存更新
            this.metaProjectRepository.update(entity);
        });
    }

    @Override
    public void dispatch(BatchLogicalDeleteProjectCommand command) {
        command.validate();

        this.unitWork.executeTran(() -> {
            for (String id : command.getIds()) {
                // 根据ID查询现有实体（包括已删除的）
                MetaProjectEntity entity = getByIdIncludeDeleted(id);
                if (entity == null) {
                    throw new NoFoundTenantException();
                }

                // 检查项目是否已删除
                if (entity.isDeleted()) {
                    throw new NoFoundTenantException();
                }

                // 标记为已删除
                TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
                entity.changeMarkAsDeleted(userModel);

                // 保存更新
                this.metaProjectRepository.update(entity);
            }
        });
    }

    @Override
    public EntityCollection<MetaProjectEntity> search(int pageIndex, int pageSize) {
        return this.metaProjectRepository.search(pageIndex, pageSize);
    }

    @Override
    public MetaProjectEntity getById(String id) {
        MetaProjectEntity entity = this.metaProjectRepository.getById(id);
        
        // 如果记录已被逻辑删除，返回null
        if (entity != null && entity.isDeleted()) {
            return null;
        }
        
        return entity;
    }

    /**
     * 获取项目实体（包括已删除的记录）
     * 仅供内部使用
     */
    private MetaProjectEntity getByIdIncludeDeleted(String id) {
        return this.metaProjectRepository.getById(id);
    }
}
