package com.bcxin.tenant.bcx.domains.services.impls;

import com.bcxin.tenant.bcx.domains.entities.MetaMenuEntity;
import com.bcxin.tenant.bcx.domains.repositories.MetaMenuRepository;
import com.bcxin.tenant.bcx.domains.repositories.criterias.MetaMenuSearchCriteria;
import com.bcxin.tenant.bcx.domains.services.MetaMenuService;
import com.bcxin.tenant.bcx.domains.services.commands.DeleteDataCommand;
import com.bcxin.tenant.bcx.domains.services.commands.menus.CreateMetaMenuCommand;
import com.bcxin.tenant.bcx.domains.services.commands.menus.UpdateMetaMenuCommand;
import com.bcxin.tenant.bcx.infrastructures.EntityCollection;
import com.bcxin.tenant.bcx.infrastructures.components.IdWorker;
import com.bcxin.tenant.bcx.infrastructures.UnitWork;
import com.bcxin.tenant.bcx.infrastructures.exceptions.NoFoundTenantException;
import com.bcxin.tenant.bcx.infrastructures.utils.StringUtil;
import org.springframework.stereotype.Service;

import java.sql.Timestamp;

/**
 * 菜单元数据服务实现
 */
@Service
public class MetaMenuServiceImpl implements MetaMenuService {
    private final MetaMenuRepository menuRepository;
    private final UnitWork unitWork;
    private final IdWorker idWorker;

    public MetaMenuServiceImpl(MetaMenuRepository menuRepository, UnitWork unitWork, IdWorker idWorker) {
        this.menuRepository = menuRepository;
        this.unitWork = unitWork;
        this.idWorker = idWorker;
    }

    @Override
    public void dispatch(CreateMetaMenuCommand command) {
        command.validate();

        if(StringUtil.isEmpty(command.getName())){
            throw new IllegalArgumentException("菜单名称不能为空");
        }

        this.unitWork.executeTran(() -> {
            String id = String.valueOf(this.idWorker.getNextId());
            MetaMenuEntity entity = new MetaMenuEntity();
            entity.setId(id);
            entity.setProjectId(command.getProjectId());
            entity.setName(command.getName());
            entity.setPageId(command.getPageId());
            entity.setParentId(command.getParentId());
            entity.setIndexTreeId(command.getIndexTreeId());
            entity.setSlug(command.getSlug());
            entity.setMediumType(command.getMediumType());
            
            Timestamp now = new Timestamp(System.currentTimeMillis());
            entity.setCreatedTime(now);
            entity.setLastUpdatedTime(now);
            this.menuRepository.insert(entity);
        });
    }

    @Override
    public void dispatch(UpdateMetaMenuCommand command) {
        command.validate();
        this.unitWork.executeTran(() -> {
            MetaMenuEntity entity = this.menuRepository.getById(command.getId());
            if (entity == null) {
                throw new NoFoundTenantException();
            }

            // 更新字段
            if (command.getName() != null) {
                entity.setName(command.getName());
            }
            if (command.getPageId() != null) {
                entity.setPageId(command.getPageId());
            }
            if (command.getParentId() != null) {
                entity.setParentId(command.getParentId());
            }
            if (command.getIndexTreeId() != null) {
                entity.setIndexTreeId(command.getIndexTreeId());
            }
            if (command.getSlug() != null) {
                entity.setSlug(command.getSlug());
            }
            if (command.getMediumType() != null) {
                entity.setMediumType(command.getMediumType());
            }

            entity.setLastUpdatedTime(new Timestamp(System.currentTimeMillis()));
            this.menuRepository.update(entity);
        });
    }

    @Override
    public void dispatch(DeleteDataCommand command) {
        command.validate();
        this.unitWork.executeTran(() -> {
            MetaMenuEntity entity = this.menuRepository.getById(command.getId());
            if (entity == null) {
                throw new NoFoundTenantException();
            }
            this.menuRepository.delete(command.getId());
        });
    }

    @Override
    public EntityCollection<MetaMenuEntity> search(MetaMenuSearchCriteria criteria) {
        return this.menuRepository.search(criteria);
    }

    @Override
    public MetaMenuEntity getById(String id) {
        return this.menuRepository.getById(id);
    }
}
