package com.bcxin.tenant.bcx.domains.repositories;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.bcxin.tenant.bcx.domains.components.CacheProvider;
import com.bcxin.tenant.bcx.domains.entities.MetaPageEntity;
import com.bcxin.tenant.bcx.domains.mappers.MetaPageMapper;
import com.bcxin.tenant.bcx.domains.repositories.criterias.MetaPageSearchCriteria;
import com.bcxin.tenant.bcx.infrastructures.EntityCollection;
import com.bcxin.tenant.bcx.infrastructures.utils.StringUtil;
import org.springframework.stereotype.Repository;

import java.sql.Timestamp;
import java.time.Instant;
import java.util.Collections;
import java.util.List;

/**
 * 页面元数据仓库实现
 */
@Repository
public class MetaPageRepositoryImpl extends RepositoryAbstract<MetaPageEntity> implements MetaPageRepository {
    private final MetaPageMapper mapper;
    private final CacheProvider cacheProvider;

    protected MetaPageRepositoryImpl(MetaPageMapper mapper, CacheProvider cacheProvider) {
        super(mapper);
        this.mapper = mapper;
        this.cacheProvider = cacheProvider;
    }

    @Override
    public EntityCollection<MetaPageEntity> search(MetaPageSearchCriteria criteria) {
        // 创建 Page 对象
        Page<MetaPageEntity> page = new Page<>(criteria.getPageIndex(), criteria.getPageSize());

        // 使用 LambdaQueryWrapper 构建查询条件
        LambdaQueryWrapper<MetaPageEntity> queryWrapper = new LambdaQueryWrapper<>();
        
        // deleted = 0
        queryWrapper.eq(MetaPageEntity::isDeleted, false);
        
        // id = #{criteria.id}
        if (!StringUtil.isEmpty(criteria.getId())) {
            queryWrapper.eq(MetaPageEntity::getId, criteria.getId());
        }
        
        // id IN (...)
        List<String> ids = criteria.getIds();
        if (ids != null && !ids.isEmpty()) {
            queryWrapper.in(MetaPageEntity::getId, ids);
        }
        
        // project_id = #{criteria.projectId}
        if (!StringUtil.isEmpty(criteria.getProjectId())) {
            queryWrapper.eq(MetaPageEntity::getProjectId, criteria.getProjectId());
        }
        
        // module_id = #{criteria.moduleId}
        if (!StringUtil.isEmpty(criteria.getModuleId())) {
            queryWrapper.eq(MetaPageEntity::getModuleId, criteria.getModuleId());
        }
        
        // name LIKE CONCAT('%',#{criteria.name},'%')
        if (!StringUtil.isEmpty(criteria.getName())) {
            queryWrapper.like(MetaPageEntity::getName, criteria.getName());
        }
        
        // slug = #{criteria.slug}
        if (!StringUtil.isEmpty(criteria.getSlug())) {
            queryWrapper.eq(MetaPageEntity::getSlug, criteria.getSlug());
        }
        
        // page_type = #{criteria.pageType} OR page_type IN (...)
        if (criteria.getPageTypes() != null && !criteria.getPageTypes().isEmpty()) {
            // 多选：使用 IN 查询
            List<Integer> pageTypeOrdinals = criteria.getPageTypes().stream()
                    .map(Enum::ordinal)
                    .collect(java.util.stream.Collectors.toList());
            queryWrapper.in(MetaPageEntity::getPageType, pageTypeOrdinals);
        } else if (criteria.getPageType() != null) {
            // 单选：使用 EQ 查询（向后兼容）
            queryWrapper.eq(MetaPageEntity::getPageType, criteria.getPageType().ordinal());
        }
        
        // business_type = #{criteria.businessType} OR business_type IN (...)
        if (criteria.getBusinessTypes() != null && !criteria.getBusinessTypes().isEmpty()) {
            // 多选：使用 IN 查询
            List<Integer> businessTypeOrdinals = criteria.getBusinessTypes().stream()
                    .map(Enum::ordinal)
                    .collect(java.util.stream.Collectors.toList());
            queryWrapper.in(MetaPageEntity::getBusinessType, businessTypeOrdinals);
        } else if (criteria.getBusinessType() != null) {
            // 单选：使用 EQ 查询（向后兼容）
            queryWrapper.eq(MetaPageEntity::getBusinessType, criteria.getBusinessType().ordinal());
        }
        
        // table_name LIKE CONCAT('%',#{criteria.tableName},'%')
        if (!StringUtil.isEmpty(criteria.getTableName())) {
            queryWrapper.like(MetaPageEntity::getTableName, criteria.getTableName());
        }
        
        // enable_bpm_flow = #{criteria.enableBPMFlow}
        if (criteria.getEnableBPMFlow() != null) {
            queryWrapper.eq(MetaPageEntity::isEnableBPMFlow, criteria.getEnableBPMFlow());
        }
        
        // ORDER BY created_time DESC
        queryWrapper.orderByDesc(MetaPageEntity::getCreatedTime);

        // 使用 mapper 查询数据
        IPage<MetaPageEntity> result = mapper.selectPage(page, queryWrapper);

        // 转换为 EntityCollection
        return EntityCollection.create(
                result.getRecords(),
                criteria.getPageSize(),
                result.getTotal()
        );
    }

    @Override
    public List<MetaPageEntity> getByIds(List<String> ids) {
        if (ids == null || ids.isEmpty()) {
            return Collections.emptyList();
        }
        return this.mapper.selectBatchIds(ids);
    }

    @Override
    public void batchMarkAsDeleted(List<String> ids, Timestamp updateTime) {
        if (ids == null || ids.isEmpty()) {
            return;
        }
        
        LambdaUpdateWrapper<MetaPageEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.in(MetaPageEntity::getId, ids)
                     .eq(MetaPageEntity::isDeleted, false)
                     .set(MetaPageEntity::isDeleted, true)
                     .set(MetaPageEntity::getLastUpdatedTime, updateTime);
        
        this.mapper.update(null, updateWrapper);
    }

    @Override
    public void delete(Object id) {
        LambdaUpdateWrapper<MetaPageEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.eq(MetaPageEntity::getId, id)
                .eq(MetaPageEntity::isDeleted, false)
                .set(MetaPageEntity::isDeleted, true)
                .set(MetaPageEntity::getLastUpdatedTime, Timestamp.from(Instant.now()));

        this.mapper.update(null, updateWrapper);
    }

    @Override
    public MetaPageEntity getById(Object id) {
        return super.getById(id);
    }

    @Override
    public void update(MetaPageEntity data) {
        super.update(data);
    }
}

