package com.bcxin.tenant.bcx.domains.repositories;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.bcxin.tenant.bcx.domains.components.CacheProvider;
import com.bcxin.tenant.bcx.domains.criterias.DynamicPageDataCriteria;
import com.bcxin.tenant.bcx.domains.entities.DynamicPageDataEntity;
import com.bcxin.tenant.bcx.domains.mappers.DynamicPageDataMapper;
import com.bcxin.tenant.bcx.domains.utils.DynamicPageDataEntityUtils;
import com.bcxin.tenant.bcx.domains.wrappers.DynamicDataWrapper;
import com.bcxin.tenant.bcx.infrastructures.EntityCollection;
import com.bcxin.tenant.bcx.infrastructures.TenantContext;
import com.bcxin.tenant.bcx.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.bcx.infrastructures.components.JsonProvider;
import com.bcxin.tenant.bcx.infrastructures.enums.DoActionType;
import com.bcxin.tenant.bcx.infrastructures.exceptions.*;
import com.bcxin.tenant.bcx.domains.snapshots.DynamicPageTableMappingDataSnapshot;
import com.bcxin.tenant.bcx.infrastructures.utils.ExceptionUtil;
import com.bcxin.tenant.bcx.infrastructures.valueTypes.DynamicConditionFieldValueType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Repository;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Repository
public class DynamicPageDataRepositoryImpl extends RepositoryAbstract<DynamicPageDataEntity>
        implements DynamicPageDataRepository {
    private static final Logger logger = LoggerFactory.getLogger(DynamicPageDataRepositoryImpl.class);

    /**
     * 通过持久化来区分不同的表内容
     */
    private final MetaPageTableMappingRepository tableMappingRepository;
    private final CacheProvider cacheProvider;
    private final DynamicPageDataMapper dynamicPageDataMapper;
    private final JsonProvider jsonProvider;
    protected DynamicPageDataRepositoryImpl(DynamicPageDataMapper mapper,
                                            MetaPageTableMappingRepository tableMappingRepository,
                                            CacheProvider cacheProvider, JsonProvider jsonProvider) {
        super(mapper);
        this.dynamicPageDataMapper = mapper;
        this.tableMappingRepository = tableMappingRepository;
        this.cacheProvider = cacheProvider;
        this.jsonProvider = jsonProvider;
    }

    @Override
    public EntityCollection<DynamicPageDataEntity> search(DynamicPageDataCriteria criteria) {

        DynamicPageTableMappingDataSnapshot snapshot = null;
        if(!StringUtils.hasLength(criteria.getMappingId()))
        {
            snapshot =this.tableMappingRepository.getSnapshot(criteria.getPageId());
        }else {
            snapshot = this.tableMappingRepository.getSnapshotByMappingId(criteria.getMappingId());
        }

        if(!DynamicPageTableMappingDataSnapshot.isEmpty(snapshot)) {
            DynamicDataWrapper wrapper =
                    DynamicDataWrapper.createForSearch(
                            snapshot, criteria.getFiledKeyValueMap(),
                            criteria.ignorePage(), criteria.getPageIndex(),
                            criteria.getPageSize());
            wrapper = getWrappedWrapper(wrapper);

            Collection<Map<String, Object>> rt =
                    this.dynamicPageDataMapper.dynamicSearch(
                            wrapper
                    );

            long count = this.dynamicPageDataMapper.dynamicCount(
                    wrapper
            );

            Collection<DynamicPageDataEntity> entities =
                    rt.stream().map(ii -> DynamicPageDataEntityUtils.translate2PageDataEntity(ii, jsonProvider)).collect(Collectors.toList());
            return EntityCollection.create(entities, criteria.getPageSize(), count);
        }else if(StringUtils.hasLength(criteria.getMappingId())) {
            throw new NoFoundTenantException(String.format("映射表单(%s)无效", criteria.getPageId()));
        }

        LambdaQueryWrapper<DynamicPageDataEntity> lambdaQueryWrapper = new LambdaQueryWrapper<>();
        lambdaQueryWrapper.eq(DynamicPageDataEntity::getPageId, criteria.getPageId());

        if (!CollectionUtils.isEmpty(criteria.getFields())) {
            for (DynamicConditionFieldValueType field : criteria.getFields()) {
                if (field.getValue() != null ) {
                    if(!StringUtils.hasLength(field.getType())) {
                        logger.warn("非预期数据;字段(page={}, field={})类型不应该为空", criteria.getPageId(), field.getField());
                        continue;
                    }

                    switch (field.getType()) {
                        case "datePicker":
                        case "timePicker":
                            List<String> values = (List<String>) field.getValue();
                            if (!CollectionUtils.isEmpty(values) && values.size() == 2) {
                                String whereSql = String.format("(data_json ->>'$.items.%s' >= {0} and data_json ->>'$.items.%s' <= {1})",
                                        field.getField(), field.getField());
                                lambdaQueryWrapper.apply(whereSql, values.get(0), values.get(1));
                            }
                            break;
                        case "inputNumber":
                            Collection<Integer> numbers = (Collection<Integer>) field.getValue();
                            lambdaQueryWrapper.apply(String.format("(data_json ->>'$.items.%s' = {0})", field.getField()), field.getValue());
                            break;
                        case "input":
                            lambdaQueryWrapper.apply(String.format("(data_json ->>'$.items.%s' like {0})", field.getField()), field.getValue() + "%");
                            break;
                        default:
                            lambdaQueryWrapper.apply(String.format("(data_json ->>'$.items.%s' = {0})", field.getField()), field.getValue());
                            break;
                    }
                }
            }
        }

        lambdaQueryWrapper.last("order by created_time desc,last_updated_time desc");
        if (criteria.isForExport()) {
            Collection<DynamicPageDataEntity> entities = this.mapper.selectList(lambdaQueryWrapper);

            return EntityCollection.create(entities,criteria.getPageSize(),(long)entities.size());
        } else {
            IPage<DynamicPageDataEntity> page = new Page<>(criteria.getPageIndex(), criteria.getPageSize());

            IPage<DynamicPageDataEntity> pageDataEntityIPage =
                    this.mapper.selectPage(page, lambdaQueryWrapper);

            return EntityCollection.create(
                    pageDataEntityIPage.getRecords(),
                    (int) pageDataEntityIPage.getSize(),
                    pageDataEntityIPage.getTotal()
            );
        }
    }

    @Override
    public DynamicPageDataEntity getById(String pageId, String id) {
        DynamicPageTableMappingDataSnapshot snapshot = this.tableMappingRepository.getSnapshot(pageId);
        if(!DynamicPageTableMappingDataSnapshot.isEmpty(snapshot)) {
            DynamicDataWrapper wrapper = DynamicDataWrapper.createForGetAndDeleteAction(snapshot, pageId, id);
            wrapper = getWrappedWrapper(wrapper);

            Map<String, Object> items =
                    this.dynamicPageDataMapper.getDslById(
                            wrapper
                    );
            if(CollectionUtils.isEmpty(items)) {
                throw new NoFoundTenantException();
            }

            return DynamicPageDataEntityUtils.translate2PageDataEntity(items, jsonProvider);
        }

        LambdaQueryWrapper<DynamicPageDataEntity> lambdaQueryWrapper = new LambdaQueryWrapper<>();
        lambdaQueryWrapper.eq(DynamicPageDataEntity::getPageId, pageId);
        lambdaQueryWrapper.eq(DynamicPageDataEntity::getId, id);

        return this.mapper.selectOne(lambdaQueryWrapper);
    }

    @Override
    public void delete(String pageId, String id) {
        DynamicPageTableMappingDataSnapshot snapshot = this.tableMappingRepository.getSnapshot(pageId);
        if (!DynamicPageTableMappingDataSnapshot.isEmpty(snapshot)) {
            int affectCount = this.dynamicPageDataMapper.dynamicDelete(
                    DynamicDataWrapper.createForGetAndDeleteAction(snapshot,pageId,id)
            );

            if(affectCount<1) {
                throw new ForbidTenantException("该数据不允许删除");
            }

            return;
        }

        LambdaQueryWrapper<DynamicPageDataEntity> lambdaQueryWrapper = new LambdaQueryWrapper<>();
        lambdaQueryWrapper.eq(DynamicPageDataEntity::getPageId, pageId);
        lambdaQueryWrapper.eq(DynamicPageDataEntity::getId, id);

        int affectedCount = this.mapper.delete(lambdaQueryWrapper);

        if (affectedCount != 1) {
            logger.warn("非预期数据;pageId={},id={}数据不存在", pageId, id);
            throw new NoFoundTenantException();
        }
    }

    @Override
    public void batchInsert(Collection<DynamicPageDataEntity> entities) {
        if (CollectionUtils.isEmpty(entities)) {
            return;
        }
        List<String> pageIds = entities.stream().map(ii -> ii.getPageId()).distinct().collect(Collectors.toList());
        if (pageIds.size() > 1) {
            throw new NotSupportTenantException("不支持同时批量处理多页面数据");
        }

        String pageId = pageIds.get(0);
        DynamicPageTableMappingDataSnapshot snapshot =
                this.tableMappingRepository.getSnapshot(pageId);
        if (!DynamicPageTableMappingDataSnapshot.isEmpty(snapshot)) {
            try {
                Collection<DynamicDataWrapper> wrappers =
                        entities.stream().map(ii -> DynamicDataWrapper.create(
                                snapshot, ii, jsonProvider, DoActionType.Create
                        )).collect(Collectors.toList());

                for(DynamicDataWrapper wrapper: wrappers) {
                    this.dynamicPageDataMapper.dynamicInsertOnDuplicateUpdate(
                            wrapper
                    );
                }
            } catch (Exception ex) {
                if (!CollectionUtils.isEmpty(snapshot.getValidationRules())) {
                    String error = ExceptionUtil.getStackMessage(ex);
                    for (String identity : snapshot.getValidationRules().keySet()) {
                        if (error.contains(identity)) {
                            throw new ConflictTenantException(String.valueOf(snapshot.getValidationRules().get(identity)));
                        }
                    }
                }

                throw new BadTenantException("数据保存异常", ex);
            }

            return;
        }

        for (var data : entities) {
            super.insert(data);
        }
    }

    @Override
    public void doAction(DynamicPageDataEntity data, DoActionType actionType) {
        DynamicPageTableMappingDataSnapshot snapshot = this.tableMappingRepository.getSnapshot(data.getPageId());
        if (!DynamicPageTableMappingDataSnapshot.isEmpty(snapshot)) {
            DynamicDataWrapper wrapper = DynamicDataWrapper.create(snapshot, data, jsonProvider, actionType);
            int affectedCount = this.dynamicPageDataMapper.dynamicDoAction(
                    wrapper
            );

            if (affectedCount < 1) {
                logger.error("无影响数据行:{} 参数={}", wrapper.getActionDsl(), wrapper.getParams());
            }

            return;
        }

        super.update(data);
    }

    @Override
    public void insert(DynamicPageDataEntity data) {
        DynamicPageTableMappingDataSnapshot snapshot =
                this.tableMappingRepository.getSnapshot(data.getPageId());
        if (!DynamicPageTableMappingDataSnapshot.isEmpty(snapshot)) {
            try {
                int affectedCount = this.dynamicPageDataMapper.dynamicInsert(
                        DynamicDataWrapper.create(snapshot, data, jsonProvider, DoActionType.Create)
                );

                if (affectedCount < 1) {
                    throw new ConflictTenantException("该数据已重复");
                }

                return;
            }
            catch (Exception ex) {
                if (!CollectionUtils.isEmpty(snapshot.getValidationRules())) {
                    String error = ExceptionUtil.getStackMessage(ex);
                    for (String identity : snapshot.getValidationRules().keySet()) {
                        if (error.contains(identity)) {
                            throw new ConflictTenantException(String.valueOf(snapshot.getValidationRules().get(identity)));
                        }
                    }
                }

                throw new BadTenantException("数据保存异常", ex);
            }
        }

        super.insert(data);
    }

    @Override
    public void update(DynamicPageDataEntity data) {
        DynamicPageTableMappingDataSnapshot snapshot = this.tableMappingRepository.getSnapshot(data.getPageId());
        if (!DynamicPageTableMappingDataSnapshot.isEmpty(snapshot)) {
            this.dynamicPageDataMapper.dynamicUpdate(
                    DynamicDataWrapper.create(snapshot, data, jsonProvider, DoActionType.Update)
            );

            return;
        }

        super.update(data);
    }

    @Override
    public void delete(Object id) {
        throw new NotSupportTenantException();
    }

    private DynamicDataWrapper getWrappedWrapper(DynamicDataWrapper wrapper){
        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if(userModel!=null) {
            wrapper.addParams(userModel.fillAndReturnMap(wrapper.getParams()));
            wrapper.assignCurrentUser(userModel);
        }

        return wrapper;
    }
}
