package com.bcxin.tenant.bcx.backend.tasks.configs;

import com.bcxin.tenant.bcx.infrastructures.constants.KafkaConstants;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerConfig;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.config.TopicConfig;
import org.apache.kafka.common.header.Headers;
import org.apache.kafka.common.header.internals.RecordHeaders;
import org.apache.kafka.common.serialization.StringDeserializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.kafka.annotation.EnableKafka;
import org.springframework.kafka.config.ConcurrentKafkaListenerContainerFactory;
import org.springframework.kafka.config.KafkaListenerContainerFactory;
import org.springframework.kafka.config.TopicBuilder;
import org.springframework.kafka.core.*;
import org.springframework.kafka.listener.*;

import java.util.HashMap;
import java.util.Map;
import java.util.function.BiFunction;

/**
 * https://www.baeldung.com/spring-kafka
 */
@EnableKafka
@Configuration
public class KafkaConfig {
    private static final Logger logger = LoggerFactory.getLogger(KafkaConfig.class);

    public KafkaConfig() {
    }

    @Value("${spring.kafka.bootstrap-servers}")
    private String bootstrapServer;

    @Value("${spring.kafka.consumer.enable-auto-commit}")
    private Boolean autoCommit;
    @Value("${spring.kafka.consumer.group-id}")private String groupId;

    @Value("${spring.kafka.consumer.auto-offset-reset}")
    private String autoOffsetReset;

    @Value("${spring.kafka.consumer.fetch-min-size}")
    private String fetchMinSize;

    @Value("${spring.kafka.consumer.fetch-max-wait}")
    private String fetchMaxWait;

    @Value("${spring.kafka.listener.ack-mode}")
    private ContainerProperties.AckMode ackMode;

    @Value("${spring.kafka.listener.concurrency}")
    private Integer concurrency;

    /**
     * 如果topic不存在, 那么则创建对应的主题
     * @return
     */
    @Bean
    public KafkaAdmin.NewTopics topics() {
        int partitionCount = 3;
        int replicaCount = 1;
        return new KafkaAdmin.NewTopics(
                /**
                 * 有限有效消息队列: 该队列只保留20分钟的数据
                 */
                TopicBuilder.name(KafkaConstants.TOPIC_DEAD_LETTER_DISPATCH)
                        .partitions(partitionCount)
                        .replicas(replicaCount)
                        .build(),
                TopicBuilder.name(KafkaConstants.HEADER_ORIGINAL_TOPIC_NAME)
                        .partitions(partitionCount)
                        .replicas(replicaCount)
                        .build()
        );
    }


    @Bean
    KafkaListenerContainerFactory<ConcurrentMessageListenerContainer<String, String>> kafkaListenerContainerFactory(
            CommonErrorHandler commonErrorHandler) {
        ConcurrentKafkaListenerContainerFactory<String, String> factory =
                new ConcurrentKafkaListenerContainerFactory<>();
        factory.setConsumerFactory(consumerFactory());
        ContainerProperties containerProperties =
                factory.getContainerProperties();
        containerProperties.setPollTimeout(36_000);
        containerProperties.setAckMode(ackMode);
        factory.setBatchListener(true);
        factory.setAutoStartup(true);
        if(this.concurrency!=null) {
            factory.setConcurrency(this.concurrency.intValue());
        }

        factory.setCommonErrorHandler(commonErrorHandler);
        factory.setMissingTopicsFatal(true);

        return factory;
    }

    @Bean
    public ConsumerFactory<String, String> consumerFactory() {
        return new DefaultKafkaConsumerFactory<>(consumerConfigs());
    }

    @Bean
    public Map<String, Object> consumerConfigs() {
        Map<String, Object> props = new HashMap<>();

        props.put(ConsumerConfig.BOOTSTRAP_SERVERS_CONFIG, bootstrapServer);
        props.put(ConsumerConfig.ENABLE_AUTO_COMMIT_CONFIG, autoCommit);
        props.put(ConsumerConfig.GROUP_ID_CONFIG, groupId);
        props.put(ConsumerConfig.KEY_DESERIALIZER_CLASS_CONFIG, StringDeserializer.class);
        props.put(ConsumerConfig.VALUE_DESERIALIZER_CLASS_CONFIG, StringDeserializer.class);
        props.put(ConsumerConfig.AUTO_OFFSET_RESET_CONFIG, autoOffsetReset);
        props.put(ConsumerConfig.FETCH_MIN_BYTES_CONFIG, this.fetchMinSize);
        props.put(ConsumerConfig.FETCH_MAX_WAIT_MS_CONFIG, this.fetchMaxWait);

        /**
         * https://www.coder.work/article/7748413
         */
        props.put(ConsumerConfig.REQUEST_TIMEOUT_MS_CONFIG, "60000");
        props.put(ConsumerConfig.FETCH_MAX_WAIT_MS_CONFIG, 3000);

        return props;
    }

    @Bean
    public CommonErrorHandler commonErrorHandler(KafkaTemplate kafkaTemplate) {
        final BiFunction<ConsumerRecord<?, ?>, Exception, TopicPartition>
                destinationResolver = (cr, e) -> new TopicPartition(KafkaConstants.TOPIC_DEAD_LETTER_DISPATCH, cr.partition());

        BiFunction<ConsumerRecord<?, ?>, Exception, Headers> DEFAULT_HEADERS_FUNCTION =
                (rec, ex) -> {
                    Headers topicNameHeader = new RecordHeaders();
                    topicNameHeader.add(KafkaConstants.HEADER_ORIGINAL_TOPIC_NAME, rec.topic().getBytes());

                    return topicNameHeader;
                };
        DeadLetterPublishingRecoverer deadLetterPublishingRecoverer = new DeadLetterPublishingRecoverer(kafkaTemplate, destinationResolver) {
            @Override
            public void accept(ConsumerRecord<?, ?> record, Consumer<?, ?> consumer, Exception exception) {
                boolean disallowedOrException = true;
                try {
                    if (!KafkaConstants.TOPIC_DEAD_LETTER_DISPATCH.equalsIgnoreCase(record.topic())) {
                        super.accept(record, consumer, exception);
                        disallowedOrException = false;
                    }
                } catch (Exception ex) {
                    disallowedOrException = true;
                }

                if (disallowedOrException) {
                    logger.error(String.format("send data to dead letter queue while error:topic=%s;key=%s;", record.topic(), record.key()));
                }
            }
        };

        deadLetterPublishingRecoverer.addHeadersFunction(DEFAULT_HEADERS_FUNCTION);
        return new DefaultErrorHandler(deadLetterPublishingRecoverer);
    }

    @Bean
    public CloseableHttpClient httpClient(){
        return HttpClientBuilder.create().build();
    }
}
