package com.bcxin.backend.core.components;

import com.bcxin.backend.core.models.CacheItemAbstract;

import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;

public interface CacheProvider {
    <T> T getData(String key, Supplier<T> supplier);

    void reset(String key);

    <T> void update(String key, T data);

    static class CacheProviderImpl implements CacheProvider
    {
        private static ConcurrentHashMap<String, Object> concurrentHashMap = new ConcurrentHashMap<>();
        @Override
        public <T> T getData(String key, Supplier<T> supplier) {
            assert supplier != null;

            Object data = concurrentHashMap.get(key);
            if (data == null) {
                data = supplier.get();
                update(key, data);
            }

            if (data instanceof CacheItemAbstract && ((CacheItemAbstract) data).isExpired()) {
                data = supplier.get();
                update(key, data);
            }

            return (T) data;
        }

        @Override
        public void reset(String key) {
            concurrentHashMap.remove(key);
        }

        @Override
        public <T> void update(String key, T data) {
            if (data == null) {
                if (concurrentHashMap.contains(key)) {
                    concurrentHashMap.remove(key);
                }
                return;
            }

            concurrentHashMap.put(key, data);
        }
    }
}
