package com.bcxin.flink.streaming.cores;
import com.bcxin.event.core.FlinkConstants;
import org.apache.commons.lang3.StringUtils;
import org.apache.flink.api.common.restartstrategy.RestartStrategies;
import org.apache.flink.api.common.time.Time;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.configuration.MemorySize;
import org.apache.flink.configuration.TaskManagerOptions;
import org.apache.flink.streaming.api.CheckpointingMode;
import org.apache.flink.streaming.api.environment.CheckpointConfig;
import org.apache.flink.streaming.api.environment.StreamExecutionEnvironment;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;
import java.util.concurrent.TimeUnit;

public class StreamingCoreEnvironments {
    private final static Logger logger = LoggerFactory.getLogger(StreamingCoreEnvironments.class);

    /**
     * 获取基本的配置以及上次的savepoint点
     * https://cloud.tencent.com/developer/article/2010848
     * 销售，
     * 这边使用hdfs的一个问题是, 会导致datanode的节点挂掉问题
     *
     * @return
     */
    public static StreamExecutionEnvironment getStreamExecutionEnvironment(CoreJobContext jobContext, boolean enableCheckPoint) {
        logger.info("开始获取getStreamExecutionEnvironment对象:SavepointPath={};CheckpointPath={}",
                jobContext.getConfigProperty().getSavepointPath(),
                jobContext.getConfigProperty().getCheckpointPath());
        Configuration configuration = new Configuration();
        /**
         * TaskExecutorResourceUtils.adjustForLocalExecution
         * 设置默认值
         */
        Map<String, String> flinkConfig = jobContext.getFlinkConfig();
        StringBuilder params = new StringBuilder();
        if (flinkConfig != null) {
            if (flinkConfig.containsKey(TaskManagerOptions.NETWORK_MEMORY_MIN.key())) {
                String networkMin = flinkConfig.get(TaskManagerOptions.NETWORK_MEMORY_MIN.key());
                configuration.set(TaskManagerOptions.NETWORK_MEMORY_MIN, MemorySize.parse(networkMin));
                params.append(String.format("taskmanager.memory.network.min=%s;", networkMin));
            }

            if (flinkConfig.containsKey(TaskManagerOptions.NETWORK_MEMORY_MAX.key())) {
                String networkMax = flinkConfig.get(TaskManagerOptions.NETWORK_MEMORY_MAX.key());
                configuration.set(TaskManagerOptions.NETWORK_MEMORY_MAX, MemorySize.parse(networkMax));
                params.append(String.format("taskmanager.memory.network.max=%s;", networkMax));
            }
        } else {
            configuration.set(TaskManagerOptions.NETWORK_MEMORY_MIN, MemorySize.parse("356m"));
            configuration.set(TaskManagerOptions.NETWORK_MEMORY_MAX, MemorySize.parse("356m"));
            params.append("设置默认值:=356m");
        }

        logger.error("(无配置NETWORK-启用v2.Checkpoint={}), 传递的flink参数信息:{};TaskManagerOptions.NETWORK_MEMORY_MAX={}",
                enableCheckPoint, params,
                configuration.get(TaskManagerOptions.NETWORK_MEMORY_MAX).getMebiBytes()
        );
        configuration = new Configuration();
        if (enableCheckPoint) {
            if (!StringUtils.isEmpty(jobContext.getConfigProperty().getSavepointPath())) {
                configuration.setString("execution.savepoint.path", jobContext.getConfigProperty().getSavepointPath());
            }

            StreamExecutionEnvironment env = StreamExecutionEnvironment.getExecutionEnvironment(configuration);
            env.setRestartStrategy(
                    RestartStrategies.fixedDelayRestart(10_000,
                            Time.of(5, TimeUnit.SECONDS))
            );
            env.enableCheckpointing(100, CheckpointingMode.EXACTLY_ONCE);

            CheckpointConfig checkpointConfig = env.getCheckpointConfig();

            checkpointConfig.setCheckpointStorage(jobContext.getConfigProperty().getCheckpointPath());
            checkpointConfig.setCheckpointingMode(CheckpointingMode.EXACTLY_ONCE);
            checkpointConfig.setMaxConcurrentCheckpoints(2);

            return env;
        }

        StreamExecutionEnvironment tableEnv = StreamExecutionEnvironment.getExecutionEnvironment(configuration);
        tableEnv.setRestartStrategy(
                RestartStrategies.fixedDelayRestart(10_000,
                        Time.of(5, TimeUnit.SECONDS))
        );

        return tableEnv;
    }

    /*
        //设置后端状态
    private static void setStateBackend(StreamExecutionEnvironment env,
                                        CheckPointParam checkPointParam) throws IOException {
        switch (checkPointParam.getStateBackendEnum()) {
            case MEMORY:
                log.info("开启MEMORY模式");
                env.setStateBackend(
                        new MemoryStateBackend(MemoryStateBackend.DEFAULT_MAX_STATE_SIZE * 100));
                break;
            case FILE:
                log.info("开启FILE模式");
                if (checkPointParam.getAsynchronousSnapshots() != null) {
                    env.setStateBackend(new FsStateBackend(checkPointParam.getCheckpointDir(),
                            checkPointParam.getAsynchronousSnapshots()));
                } else {
                    env.setStateBackend(new FsStateBackend(checkPointParam.getCheckpointDir()));
                }
                break;
            case ROCKSDB:
                log.info("开启ROCKSDB模式");
                if (checkPointParam.getEnableIncremental() != null) {
                    env.setStateBackend(new RocksDBStateBackend(checkPointParam.getCheckpointDir(),
                            checkPointParam.getEnableIncremental()));
                } else {
                    env.setStateBackend(new RocksDBStateBackend(checkPointParam.getCheckpointDir()));
                }
                break;
            default:
                throw new RuntimeException("不支持这种后端状态" + checkPointParam.getStateBackendEnum());

        }

     */
}
