package com.bcxin.tenant.data.etc.table.tasks.components.httpsink;

import com.alibaba.fastjson.JSONObject;
import com.bcxin.event.core.JsonProvider;
import com.bcxin.event.core.JsonProviderImpl;
import com.bcxin.event.core.exceptions.BadEventException;
import com.bcxin.tenant.data.etc.table.tasks.utils.JwtUtil;
import com.bcxin.tenant.data.etc.table.tasks.webhookConfigs.WebHookConfigDefinition;
import com.bcxin.tenant.data.etc.table.tasks.webhookConfigs.WebHookConfigSourceDefinition;
import com.ctc.wstx.util.ExceptionUtil;
import lombok.Getter;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.flink.api.connector.sink2.SinkWriter;
import org.apache.http.HttpStatus;
import org.apache.http.StatusLine;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.AbstractHttpEntity;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 负责用于实现符合写入的功能
 * 业务逻辑:
 * 1, 计算数据并写入Redis
 * 2, 根据结算结果额外(原始数据 + 新kafka)产生数据写入kafka
 * @param
 */
@Getter
public class WebHttpSinkWriter implements SinkWriter<String> {
    private static final Logger logger = LoggerFactory.getLogger(WebHttpSinkWriter.class);
    private final List<String> bulkRequests = new ArrayList<>();
    private transient CloseableHttpClient httpClient = null;
    private final int batchSize;
    private final int batchIntervalMs;
    private final JsonProvider jsonProvider;
    private final WebHookConfigSourceDefinition configSourceDefinition;

    public WebHttpSinkWriter(int batchSize, int batchIntervalMs,
                             WebHookConfigSourceDefinition configSourceDefinition) {
        this.configSourceDefinition = configSourceDefinition;
        if (batchSize < 10) {
            this.batchSize = 500;
        } else {
            this.batchSize = batchSize;
        }

        if (batchIntervalMs < 500) {
            this.batchIntervalMs = 5000;
        } else {
            this.batchIntervalMs = batchIntervalMs;
        }

        this.jsonProvider = new JsonProviderImpl();

        ScheduledExecutorService executorService =
                Executors.newSingleThreadScheduledExecutor();
        executorService.scheduleAtFixedRate(() -> {
            try {
                this.flush(true);
            } catch (Exception ex) {
                ex.printStackTrace();

                logger.error("执行数据推送发生异常", ex);
            }

        }, batchIntervalMs, batchIntervalMs, TimeUnit.MICROSECONDS);

        this.httpClient = HttpClients.createDefault();
    }


    @Override
    public synchronized void write(String element, Context context) throws IOException, InterruptedException {
        bulkRequests.add(element);
        if ((batchSize > 0 && batchSize <= bulkRequests.size()) || true) {
            flush(true);
        }
    }

    @Override
    public synchronized void flush(boolean endOfInput) {
        if (CollectionUtils.isEmpty(bulkRequests)) {
            return;
        }

        try {
            String api = this.configSourceDefinition.getConf();
            this.doExecute(api, this.bulkRequests);
            this.bulkRequests.clear();
            logger.error("成功提交完毕并清楚bulkRequests={}", bulkRequests.size());
        } catch (Exception ex) {
            throw new BadEventException("执行Http Sink发生异常", ex);
        }
    }

    @Override
    public void close() throws Exception {
        this.httpClient.close();
    }

    private void doExecute(String api, List<String> parameters) throws IOException {
        String body = null;
        try {
            Collection<JSONObject> params =
                    parameters.stream().map(ii -> {
                        JSONObject data = this.jsonProvider.toObject(JSONObject.class, ii);

                        return data;
                    }).collect(Collectors.toList());

            body = this.jsonProvider.getJson(params);

            HttpPost post = new HttpPost(api);
            AbstractHttpEntity httpEntity = new StringEntity(body, "UTF-8");
            httpEntity.setContentEncoding(StandardCharsets.UTF_8.toString());
            httpEntity.setContentType("application/json");
            if (!CollectionUtils.isEmpty(configSourceDefinition.getExtendMap())) {
                for (String key : configSourceDefinition.getExtendMap().keySet()) {
                    String headValue = configSourceDefinition.getMapValue(key);
                    if (WebHookConfigSourceDefinition.CURRENT_REQUEST_USER_ID.equalsIgnoreCase(key)) {
                        post.setHeader(WebHookConfigSourceDefinition.REQUEST_HEADER_ACCESS_TOKEN_ORIGINAL, JwtUtil.getToken(headValue));
                    }
                }
            }

            post.setEntity(httpEntity);

            try (CloseableHttpResponse response = httpClient.execute(post)) {
                StatusLine statusLine = response.getStatusLine();

                if (statusLine.getStatusCode() != HttpStatus.SC_OK) {
                    String content = EntityUtils.toString(response.getEntity());
                    logger.error("执行结束(总数量={}):{} 失败-状态:{} 请求-{}; 响应:{}",
                            bulkRequests.size(),
                            api,
                            statusLine.getStatusCode(),
                            body,
                            content);
                } else {
                    logger.error("成功-执行结束(总数量={}):{} 请求-{} Ok-响应:{}",
                            bulkRequests.size(),
                            api, body, statusLine.getStatusCode());
                }
            }
        } catch (Exception ex) {
            logger.error("WebHttpSink调用(body={})发生异常:{}", body, ExceptionUtils.getMessage(ex));
            throw ex;
        }
    }
}
