package com.bcxin.tenant.data.etc.table.tasks.components.httpsink;

import org.apache.flink.annotation.Internal;
import org.apache.flink.api.common.io.RichOutputFormat;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.util.concurrent.ExecutorThreadFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import static org.apache.flink.util.Preconditions.checkNotNull;

@Internal
public class HttpOutputStringFormat extends RichOutputFormat<String> {
    private static final Logger logger = LoggerFactory.getLogger(HttpOutputStringFormat.class);

    private final HttpExecutionOptions executionOptions;

    private transient ScheduledExecutorService scheduler;
    private transient ScheduledFuture<?> scheduledFuture;
    private transient volatile boolean closed = false;
    private transient volatile Exception flushException;

    private final Collection<String> records = new ArrayList<>();

    public HttpOutputStringFormat(HttpExecutionOptions executionOptions) {
        this.executionOptions = checkNotNull(executionOptions);
    }

    @Override
    public void open(InitializationContext context) throws IOException {
        super.open(context);
        this.scheduler =
                Executors.newScheduledThreadPool(
                        1, new ExecutorThreadFactory("jdbc-upsert-output-format"));
        this.scheduledFuture =
                this.scheduler.scheduleWithFixedDelay(
                        () -> {
                            synchronized (HttpOutputStringFormat.this) {
                                if (!closed) {
                                    try {
                                        flush();
                                    } catch (Exception e) {
                                        flushException = e;
                                    }
                                }
                            }
                        },
                        this.executionOptions.getBatchIntervalMs(),
                        this.executionOptions.getBatchIntervalMs(),
                        TimeUnit.MILLISECONDS);
    }

    @Override
    public void configure(Configuration parameters) {
    }

    @Override
    public synchronized void writeRecord(String record) throws IOException {
        checkFlushException();
        this.records.add(record);

        if (record.length() >= this.executionOptions.getBatchSize()) {
            this.flush();
        }
    }

    @Override
    public void close() throws IOException {
        if (closed) {
            this.closed = true;
        }

        if (this.scheduledFuture != null) {
            scheduledFuture.cancel(false);
            this.scheduler.shutdown();
        }

        if (CollectionUtils.isEmpty(this.records)) {
            try {
                flush();
            } catch (Exception e) {
                logger.warn("Writing records to JDBC failed.", e);
                throw new RuntimeException("Writing records to JDBC failed.", e);
            }
        }
    }

    private void checkFlushException() {
        if (flushException != null) {
            throw new RuntimeException("Writing records to JDBC failed.", flushException);
        }
    }

    private void flush() {

        this.records.clear();
    }
}
