package com.bcxin.tenant.data.etc.table.tasks;

import com.bcxin.event.core.JsonProvider;
import com.bcxin.event.core.JsonProviderImpl;
import com.bcxin.event.core.exceptions.BadEventException;
import com.bcxin.event.core.utils.PropertyUtils;
import com.bcxin.flink.streaming.cores.utils.StorageUtil;
import com.bcxin.tenant.data.etc.table.tasks.components.ExtractTableFieldFromDb;
import com.bcxin.tenant.data.etc.table.tasks.components.ExtractV5MappingIdFromAccountFunction;
import com.bcxin.tenant.data.etc.table.tasks.components.httpsink.CustomHttpPostRequestCallback;
import com.bcxin.tenant.data.etc.table.tasks.components.httpsink.WebHttpSinkWriter;
import com.bcxin.tenant.data.etc.table.tasks.utils.HookConfigParseUtils;
import com.bcxin.tenant.data.etc.table.tasks.utils.JwtUtil;
import com.bcxin.tenant.data.etc.table.tasks.webhookConfigs.WebHookConfigDefinition;
import com.bcxin.tenant.data.etc.table.tasks.webhookConfigs.WebHookConfigSourceDefinition;
import com.bcxin.tenant.data.etc.table.tasks.webhookConfigs.WebHookConfigSourceType;
import com.getindata.connectors.http.HttpSink;
import com.getindata.connectors.http.HttpSinkBuilder;
import com.getindata.connectors.http.SchemaLifecycleAwareElementConverter;
import com.getindata.connectors.http.internal.sink.HttpSinkRequestEntry;
import org.apache.flink.api.connector.sink2.Sink;
import org.apache.flink.api.connector.sink2.SinkWriter;
import org.apache.flink.api.java.utils.ParameterTool;
import org.apache.flink.configuration.*;
import org.apache.flink.contrib.streaming.state.EmbeddedRocksDBStateBackend;
import org.apache.flink.streaming.api.CheckpointingMode;
import org.apache.flink.streaming.api.datastream.DataStream;
import org.apache.flink.streaming.api.environment.CheckpointConfig;
import org.apache.flink.streaming.api.environment.StreamExecutionEnvironment;
import org.apache.flink.streaming.api.functions.sink.RichSinkFunction;
import org.apache.flink.table.api.Table;
import org.apache.flink.table.api.bridge.java.StreamTableEnvironment;
import org.apache.flink.types.RowKind;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class EtcTableApp {
    private static final Logger logger = LoggerFactory.getLogger(EtcTableApp.class);

    public static void main(String[] args) throws Exception {
        if (args.length < 2) {
            throw new BadEventException("etc-table-必须传递环境信息和轨迹配置文件包名称(configs/环境/底下的文件名)");
        }

        ParameterTool parameterTool = ParameterTool.fromArgs(args);

        String env = parameterTool.get("env");//args[0];//parameterTool.get("env");
        String configFile = parameterTool.get("configFile");//args[1];//parameterTool.get("configFile");

        PropertyUtils.loadProperties(String.format("configs/%s/db.properties",env));
        Configuration configuration = new Configuration();
        configuration.set(TaskManagerOptions.MANAGED_MEMORY_SIZE, MemorySize.ofMebiBytes(512));
        if(parameterTool!=null) {
            List<ConfigOption> configOptions = Stream.of(
                    JobManagerOptions.PORT,
                    JobManagerOptions.TOTAL_PROCESS_MEMORY,
                    TaskManagerOptions.TOTAL_PROCESS_MEMORY,
                    TaskManagerOptions.NUM_TASK_SLOTS,
                    TaskManagerOptions.TASK_OFF_HEAP_MEMORY,
                    TaskManagerOptions.TASK_HEAP_MEMORY,
                    TaskManagerOptions.JVM_METASPACE,
                    TaskManagerOptions.MANAGED_MEMORY_SIZE,
                    TaskManagerOptions.MANAGED_MEMORY_SIZE,
                    RestOptions.PORT
            ).collect(Collectors.toList());

            for (ConfigOption selectedConfigOption : configOptions) {
                String optionValue = parameterTool.get(selectedConfigOption.key());
                if (StringUtils.hasLength(optionValue)) {
                    configuration.set(selectedConfigOption, Integer.parseInt(optionValue));
                }
            }
        }

        StreamExecutionEnvironment executionEnvironment = StreamExecutionEnvironment.getExecutionEnvironment(configuration);
        EmbeddedRocksDBStateBackend rocksDBStateBackend = new EmbeddedRocksDBStateBackend();

        String businessTypePath = "sql-webhook";
        String stateStoragePath = String.format("file:///%s", StorageUtil.getPath(String.format("%s-state", businessTypePath)));
        rocksDBStateBackend.setDbStoragePath(stateStoragePath);
        executionEnvironment.setStateBackend(rocksDBStateBackend);

        String checkpointPath = String.format("file:///%s", StorageUtil.getPath(businessTypePath));
        executionEnvironment.getCheckpointConfig().setCheckpointStorage(checkpointPath);
        logger.error("WebHook.etc.checkpoint的PointStorage位置={};", checkpointPath);
        executionEnvironment.enableCheckpointing(5_000);
        executionEnvironment.getCheckpointConfig().setCheckpointingMode(CheckpointingMode.EXACTLY_ONCE);
        executionEnvironment.getCheckpointConfig().setMinPauseBetweenCheckpoints(2000);
        executionEnvironment.getCheckpointConfig().setCheckpointTimeout(6_0000);
        executionEnvironment.getCheckpointConfig().setMaxConcurrentCheckpoints(1);
        executionEnvironment.getCheckpointConfig().setExternalizedCheckpointCleanup(
                CheckpointConfig.ExternalizedCheckpointCleanup.DELETE_ON_CANCELLATION);

        StreamTableEnvironment tEnv = StreamTableEnvironment.create(executionEnvironment);
        tEnv.createTemporaryFunction("extractV5MappingId", new ExtractV5MappingIdFromAccountFunction());
        tEnv.createTemporaryFunction("extractFieldFromDbTable", new ExtractTableFieldFromDb(System.getProperties()));

        WebHookConfigDefinition configDefinition =
                HookConfigParseUtils.parse(env, configFile);

        Collection<String> definitionSql = configDefinition.getHookConfigSourceSql();
        for (String sql : definitionSql) {
            tEnv.executeSql(sql);
        }

        String executeScript
                = configDefinition.getExecuteDefinition().getScript();

        logger.error("当前的SQL:{}", executeScript);

        /**
         * Http Sink的效果
         */
        RichSinkFunction<String> sinkFunction = generateRichSinkFunction(configDefinition);
        if (sinkFunction != null) {

            Table table = tEnv.sqlQuery(executeScript);

            JsonProvider jsonProvider = new JsonProviderImpl();
            DataStream<String> dataStream = tEnv.toChangelogStream(table).map(ii -> {
                        if (ii.getKind() == RowKind.UPDATE_BEFORE) {
                            return null;
                        }

                        Set<String> fieldNames = ii.getFieldNames(true);
                        HashMap<String, Object> data = new HashMap<>();
                        for (String fN : fieldNames) {
                            Object value = ii.getField(fN);
                            if (value != null) {
                                data.put(fN, ii.getField(fN));
                            }
                        }

                        String json = jsonProvider.getJson(data);
                        return json;
                    }).filter(ii -> ii != null)
                    .uid(String.format("data_stream_map_%s", executeScript.hashCode())).name("捕获商城的订单信息");


            dataStream.addSink(sinkFunction).name("执行WebHook的Http Sink");


            executionEnvironment.execute(String.format("WebHook-%s",configDefinition.getName()));
        } else {
            /**
             * 走SQL
             */
            tEnv.executeSql(executeScript).print();
        }

        logger.error("完成执行");
    }

    /**
     * https://code-cookbook.readthedocs.io/zh_CN/main/Blog%20Here/%5BFlink%5DFlink-connector-http.html
     * @return
     */
    private static HttpSink generateSinkFunction(WebHookConfigDefinition configDefinition) {
        Optional<WebHookConfigSourceDefinition> httpSinkConfigDefinitionOptional =
                configDefinition.getSources().stream()
                        .filter(ii -> ii.getType() == WebHookConfigSourceType.HttpSink)
                        .findFirst();
        if (!httpSinkConfigDefinitionOptional.isPresent()) {
            return null;
        }

        WebHookConfigSourceDefinition hookConfigSourceDefinition = httpSinkConfigDefinitionOptional.get();
        HttpSinkBuilder<String> sinkBuilder =
                HttpSink.<String>builder()
                        .setEndpointUrl(hookConfigSourceDefinition.getConf());
        if (hookConfigSourceDefinition.getMapValue(WebHookConfigSourceDefinition.INSERT_METHOD) != null) {
            sinkBuilder = sinkBuilder.setElementConverter(new SchemaLifecycleAwareElementConverter<String, HttpSinkRequestEntry>() {
                @Override
                public void open(Sink.InitContext context) {

                }
                @Override
                public HttpSinkRequestEntry apply(String s, SinkWriter.Context context) {
                    return new HttpSinkRequestEntry(
                            hookConfigSourceDefinition.getMapValue(WebHookConfigSourceDefinition.INSERT_METHOD),
                            s.getBytes(StandardCharsets.UTF_8));
                }
            });
        } else {
            sinkBuilder = sinkBuilder.setElementConverter(new SchemaLifecycleAwareElementConverter<String, HttpSinkRequestEntry>() {
                @Override
                public void open(Sink.InitContext context) {

                }

                @Override
                public HttpSinkRequestEntry apply(String s, SinkWriter.Context context) {
                    return new HttpSinkRequestEntry("POST", s.getBytes(StandardCharsets.UTF_8));
                }
            });
        }

        /**
         * 目前证明: gid.connector.http.sink.request-callback 这个是无效 slf4j-logger
         */
        sinkBuilder.setHttpPostRequestCallback(new CustomHttpPostRequestCallback());

        sinkBuilder = sinkBuilder
                /**
                 * 最大200条数据一个请求
                 */
                .setProperty("gid.connector.http.sink.request.batch.size", "200")
                /**
                 * 每5秒执行一次
                 */
                .setProperty("sink.flush-buffer.timeout", "5000")
                .setProperty("gid.connector.http.sink.writer.request.mode", "batch");

        if (!CollectionUtils.isEmpty(hookConfigSourceDefinition.getExtendMap())) {
            for (String key : hookConfigSourceDefinition.getExtendMap().keySet()) {
                String headValue = hookConfigSourceDefinition.getMapValue(key);
                if (WebHookConfigSourceDefinition.CURRENT_REQUEST_USER_ID.equalsIgnoreCase(key)) {
                    sinkBuilder =
                            sinkBuilder.setProperty(
                                    WebHookConfigSourceDefinition.REQUEST_HEADER_ACCESS_TOKEN,
                                    JwtUtil.getToken(headValue)
                            );
                } else {
                    sinkBuilder = sinkBuilder.setProperty(key, headValue);
                }
            }
        }

        sinkBuilder = sinkBuilder
                .setProperty(WebHookConfigSourceDefinition.REQUEST_HEADER_AUTHORIZATION,
                        String.format("Bearer %s", "xxxxx")
                );

        HttpSink<String> sink = sinkBuilder.build();

        return sink;
    }

    private static RichSinkFunction<String> generateRichSinkFunction(WebHookConfigDefinition configDefinition) {

        Optional<WebHookConfigSourceDefinition> httpSinkConfigDefinitionOptional =
                configDefinition.getSources().stream()
                        .filter(ii -> ii.getType() == WebHookConfigSourceType.HttpSink)
                        .findFirst();
        if (!httpSinkConfigDefinitionOptional.isPresent()) {
            return null;
        }

        WebHookConfigSourceDefinition webHookConfigSourceDefinition = httpSinkConfigDefinitionOptional.get();
        RichSinkFunction sinkFunction =
                new RichSinkFunction<String>() {
                    private WebHttpSinkWriter webHttpSinkWriter;

                    @Override
                    public void open(Configuration parameters) throws Exception {
                        super.open(parameters);
                        webHttpSinkWriter = new WebHttpSinkWriter(
                                5000, 10_000, webHookConfigSourceDefinition);
                    }

                    @Override
                    public void close() throws Exception {
                        this.webHttpSinkWriter.close();

                        super.close();
                    }

                    @Override
                    public void invoke(String value, Context context) throws Exception {
                        this.webHttpSinkWriter.write(value, null);
                    }
                };

        return sinkFunction;
    }


    private static Properties loadProperties(String propertyFile) throws IOException {
        Properties properties = System.getProperties();

        ClassLoader classLoader = PropertyUtils.class.getClassLoader();
        try (InputStream coreStream = classLoader.getResourceAsStream(propertyFile)) {
            properties.load(coreStream);
        }

        return properties;
    }
}
