package com.bcxin.data.event.transfer.web.apis.controllers;

import com.bcxin.data.event.transfer.web.apis.controllers.requests.BatchDataBinlogTransferRequest;
import com.bcxin.data.event.transfer.web.apis.controllers.requests.DataBinlogTransferRequest;
import com.bcxin.event.core.KafkaConstants;
import org.apache.kafka.clients.admin.*;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.errors.TopicExistsException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.ResponseEntity;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.*;
import java.util.stream.Collectors;

@RestController
@RequestMapping("/data-binlog-transfer")
public class DataBinlogTransferSinkController extends ControllerBase {

    private static final Set<String> _existsTopics = Collections.synchronizedSet(new HashSet<>());
    private static final Logger logger = LoggerFactory.getLogger(DataBinlogTransferSinkController.class);

    private final KafkaTemplate<String, String> kafkaTemplate;
    private final AdminClient adminClient;

    public DataBinlogTransferSinkController(KafkaTemplate<String, String> kafkaTemplate, AdminClient adminClient) {
        this.kafkaTemplate = kafkaTemplate;
        this.adminClient = adminClient;
    }

    @PostMapping
    public ResponseEntity post(@RequestBody BatchDataBinlogTransferRequest request) {
        this.ensureTopic(request);

        request.getItems().forEach(ri -> {
            ProducerRecord<String, String> record =
                    new ProducerRecord<>(
                            KafkaConstants.getSyncTopic(ri.getTopic()),
                            ri.getPartition(), ri.getKey(), ri.getValue());
            this.kafkaTemplate.send(record);
        });

        return this.ok();
    }


    private void ensureTopic(BatchDataBinlogTransferRequest request) {
        if (CollectionUtils.isEmpty(request.getItems())) {
            return;
        }

        Collection<DataBinlogTransferRequest> newTopicRequests
                = request.getItems().stream().filter(ix -> !_existsTopics.contains(KafkaConstants.getSyncTopic(ix.getTopic())))
                .collect(Collectors.toList());

        if (CollectionUtils.isEmpty(newTopicRequests)) {
            return;
        }

        for (DataBinlogTransferRequest ntr : newTopicRequests) {
            boolean isContinue2CreatePartition = true;
            try {
                NewTopic selectedNewTopic = new NewTopic(
                        KafkaConstants.getSyncTopic(ntr.getTopic()),
                        KafkaConstants.DEFAULT_PARTITION_COUNT,
                        (short) 1);

                CreateTopicsResult result =
                        this.adminClient
                                .createTopics(Collections.singleton(selectedNewTopic));
                result.all().get();

                _existsTopics.add(KafkaConstants.getSyncTopic(ntr.getTopic()));
            } catch (TopicExistsException ex) {
                _existsTopics.add(KafkaConstants.getSyncTopic(ntr.getTopic()));
                isContinue2CreatePartition = true;
            } catch (Exception ex) {
                if(ex.getCause() instanceof TopicExistsException)
                {
                    _existsTopics.add(KafkaConstants.getSyncTopic(ntr.getTopic()));
                    isContinue2CreatePartition = true;

                }else {
                    ex.printStackTrace();
                    isContinue2CreatePartition = false;
                }
            } finally {
                if (isContinue2CreatePartition) {
                    try {
                        Map<String, NewPartitions> newPartitions = new HashMap<>();
                        newTopicRequests.forEach(ri -> {
                            newPartitions.put(KafkaConstants.getSyncTopic(ntr.getTopic()),
                                    NewPartitions.increaseTo(KafkaConstants.DEFAULT_PARTITION_COUNT));
                        });

                        CreatePartitionsResult result2 = adminClient.createPartitions(newPartitions);
                        result2.all().get();
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }
        }
    }
}
