package com.bcxin.flink.cdc.dispatch.task;

import com.bcxin.event.core.JobParameterDTO;
import com.bcxin.event.core.JsonProvider;
import com.bcxin.event.core.JsonProviderImpl;
import com.bcxin.event.core.enums.JobType;
import com.bcxin.event.core.exceptions.BadEventException;
import com.bcxin.event.core.exceptions.IllegalArgumentEventException;
import com.bcxin.event.core.exceptions.NoFoundEventException;
import com.bcxin.flink.cdc.dispatch.task.jobs.Binlog2DispatchCdcJob;
import com.bcxin.flink.cdc.dispatch.task.proerpties.CdcDatabaseSourceProperty;
import com.bcxin.flink.cdc.dispatch.task.proerpties.CheckpointConfigPropertyBuilder;
import com.bcxin.flink.cdc.dispatch.task.proerpties.Constants;
import com.bcxin.flink.streaming.cores.JdbcJobExecutorUtil;
import com.bcxin.flink.streaming.cores.SystemPropertyUtil;
import com.bcxin.flink.streaming.cores.TableConstant;
import com.bcxin.flink.streaming.cores.properties.CheckpointConfigProperty;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.Arrays;
import java.util.Map;
import java.util.Properties;
import java.util.stream.Collectors;

public class JdbcBinlog2DispatchApp {
    private static final Logger logger = LoggerFactory.getLogger(JdbcBinlog2DispatchApp.class);
    private static final JsonProvider jsonProvider = new JsonProviderImpl();
    /**
     * https://cloud.tencent.com/developer/article/2010848
     * @param args
     * @throws Exception
     */
    public static void main(String[] args) throws Exception {
        System.setProperty("org.slf4j.simpleLogger.defaultLogLevel", "error");
        System.setProperty("ipc.maximum.response.length", "2147483200");
        logger.error("接收参数准备执行CDC的BINLOG读取:{}", Arrays.stream(args).collect(Collectors.joining(";")));
        buildJobEnvironment(args);

        Binlog2DispatchCdcJob job = new Binlog2DispatchCdcJob();
        job.execute();
    }

    private static void buildJobEnvironment(String[] args) throws IOException {
        Properties properties = loadProperties(args);


        CheckpointConfigProperty configProperty =
                CheckpointConfigPropertyBuilder.build(properties);

        JobContext.init(
                properties.getProperty(Constants.JOB_NAME),
                CdcDatabaseSourceProperty.create(properties),
                configProperty);
    }

    private static Properties loadProperties(String[] args) throws IOException {
        if (args == null || args.length < 2) {
            throw new BadEventException("无提供有效配置参数(环境以及jobId), 系统无法启动");
        }
        String env = args[0];
        Properties properties = SystemPropertyUtil.loadEnvConf(env);

        int jobId = Integer.parseInt(args[1]);
        JobParameterDTO jobParameterDTO = JdbcJobExecutorUtil.getJobParameter(jobId);
        if (jobParameterDTO == null) {
            throw new NoFoundEventException("参数无效; 无法加载对应的job参数");
        }

        if (jobParameterDTO.getJobType() != JobType.CDC_BINLOG) {
            throw new IllegalArgumentEventException(String.format("该jobType(%s)不支持BINLOG CDC处理", jobParameterDTO.getJobType()));
        }

        String corePropertyConf = jobParameterDTO.getParam1(jsonProvider);
        String databasePropertyConf = jobParameterDTO.getParam2(jsonProvider);

        Map<String, String> corePropertyConfMap = null;
        try {
            corePropertyConfMap = jsonProvider.toObject(Map.class, corePropertyConf);
        } catch (Exception ex) {
            throw new BadEventException(String.format("转化异常:%s", corePropertyConf), ex);
        }

        Map<String, String> databasePropertyConfMap = null;
        try {
            databasePropertyConfMap = jsonProvider.toObject(Map.class, databasePropertyConf);
        } catch (Exception ex) {
            throw new BadEventException(String.format("转化异常:%s", databasePropertyConf), ex);
        }

        properties.setProperty(TableConstant.PARAM_JOB_ID_NAME, String.valueOf(jobId));
        if (corePropertyConfMap != null) {
            for (String key : corePropertyConfMap.keySet()) {
                properties.setProperty(key, corePropertyConfMap.get(key));
            }
        }

        if (databasePropertyConfMap != null) {
            for (String key : databasePropertyConfMap.keySet()) {
                properties.setProperty(key, databasePropertyConfMap.get(key));
            }
        }

        properties.setProperty(TableConstant.PARAM_JOB_ID, String.valueOf(jobId));

        return properties;
    }
}
