package com.zbkj.admin.controller.merchant;


import cn.hutool.core.util.ObjectUtil;
import com.github.pagehelper.Page;
import com.github.pagehelper.PageHelper;
import com.zbkj.common.exception.CrmebException;
import com.zbkj.common.model.admin.SystemAdmin;
import com.zbkj.common.model.user.User;
import com.zbkj.common.page.CommonPage;
import com.zbkj.common.request.MerchantUserSearchRequest;
import com.zbkj.common.request.PageParamRequest;
import com.zbkj.common.response.UserAdminDetailResponse;
import com.zbkj.common.response.UserResponse;
import com.zbkj.common.response.bcx.BcxV5EmpResponse;
import com.zbkj.common.result.CommonResult;
import com.zbkj.common.utils.SecurityUtil;
import com.zbkj.service.service.SystemAdminService;
import com.zbkj.service.service.UserService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.lang.Nullable;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;

/**
 * 商户端用户控制器
 * +----------------------------------------------------------------------
 * | CRMEB [ CRMEB赋能开发者，助力企业发展 ]
 * +----------------------------------------------------------------------
 * | Copyright (c) 2016~2023 https://www.crmeb.com All rights reserved.
 * +----------------------------------------------------------------------
 * | Licensed CRMEB并不是自由软件，未经许可不能去掉CRMEB相关版权
 * +----------------------------------------------------------------------
 * | Author: CRMEB Team <admin@crmeb.com>
 * +----------------------------------------------------------------------
 */
@Slf4j
@RestController
@RequestMapping("api/admin/merchant/user")
@Api(tags = "商户端用户控制器")
@Validated
public class MerchantUserController {
    @Autowired
    private UserService userService;
    @Autowired
    private SystemAdminService systemAdminService;

    @PreAuthorize("hasAuthority('merchant:user:page:list')")
    @ApiOperation(value = "商户端用户分页列表")
    @RequestMapping(value = "/list", method = RequestMethod.GET)
    public CommonResult<CommonPage<UserResponse>> getList(@ModelAttribute @Validated MerchantUserSearchRequest request,
                                                          @ModelAttribute PageParamRequest pageParamRequest) {
        CommonPage<UserResponse> userCommonPage = CommonPage.restPage(userService.getMerchantPage(request, pageParamRequest));
        return CommonResult.success(userCommonPage);
    }

    @PreAuthorize("hasAuthority('merchant:user:detail')")
    @ApiOperation(value = "用户详情")
    @RequestMapping(value = "/detail/{id}", method = RequestMethod.GET)
    public CommonResult<UserAdminDetailResponse> detail(@PathVariable(value = "id") Integer id) {
        return CommonResult.success(userService.getAdminDetail(id));
    }

    @PreAuthorize("hasAuthority('merchant:user:page:list')")
    @ApiOperation(value = "获取V5职员列表")
    @RequestMapping(value = "/v5emp/list", method = RequestMethod.GET)
    public CommonResult<CommonPage<BcxV5EmpResponse>> getV5EmpList(@RequestParam @Nullable String name, @ModelAttribute PageParamRequest pageParamRequest) {
        SystemAdmin currentUser = SecurityUtil.getLoginUserVo().getUser();
        ArrayList<BcxV5EmpResponse> result = new ArrayList<>();
        User org = userService.getByAccount(currentUser.getOrgAccount());
        if (ObjectUtil.isNull(org)) {
            throw new CrmebException("找不到org用户：" + currentUser.getOrgAccount());
        }
        Page<User> pageUser = PageHelper.startPage(pageParamRequest.getPage(), pageParamRequest.getLimit());
        List<User> orgList = userService.getByV5EmpList(org.getId(), name);
        if (orgList == null || orgList.isEmpty()) {
            return CommonResult.success(CommonPage.restPage(CommonPage.copyPageInfo(pageUser, result)));
        }

        orgList.forEach(user -> {
            result.add(BcxV5EmpResponse.builder()
                    .empAccount(user.getAccount())
                    .empName(user.getNickname())
                    .phone(user.getPhone())
                    .build());
        });
        return CommonResult.success(CommonPage.restPage(CommonPage.copyPageInfo(pageUser, result)));
    }
}



