package com.bcxin.platform.service.meeting;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.bcxin.platform.common.core.text.Convert;
import com.bcxin.platform.common.exception.PlatFormBusinessException;
import com.bcxin.platform.common.utils.DateUtils;
import com.bcxin.platform.common.utils.IdWorker;
import com.bcxin.platform.common.utils.Result;
import com.bcxin.platform.common.utils.StringUtils;
import com.bcxin.platform.common.utils.bean.BeanUtils;
import com.bcxin.platform.domain.company.ComBaseInfo;
import com.bcxin.platform.domain.meeting.ComHuaweiMeetMatch;
import com.bcxin.platform.domain.meeting.PerHuaweiMeetMatch;
import com.bcxin.platform.dto.meeting.*;
import com.bcxin.platform.mapper.company.ComBaseInfoMapper;
import com.bcxin.platform.mapper.meeting.ComHuaweiMeetMatchMapper;
import com.bcxin.platform.mapper.meeting.PerHuaweiMeetMatchMapper;
import com.bcxin.platform.service.cache.TaskCacheService;
import com.bcxin.platform.service.common.CommonService;
import com.bcxin.platform.util.constants.HuaweiMeetConst;
import com.bcxin.platform.util.http.MD5Util;
import com.bcxin.platform.util.huawei.HttpBuildUtil;
import com.bcxin.platform.util.huawei.RestRequest;
import com.bcxin.platform.util.huawei.RestResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 企业华为云会议映射
 *
 * @author llc
 * @date 2019-12-11
 */
@Service
@Transactional("transactionManager")
public class ComHuaweiMeetMatchServiceImpl implements ComHuaweiMeetMatchService {

    /**
     * 日志对象
     */
    protected Logger logger = LoggerFactory.getLogger(getClass());

    @Resource
    private PerHuaweiMeetMatchMapper perHuaweiMeetMatchMapper;

    @Resource
    private ComHuaweiMeetMatchMapper comHuaweiMeetMatchMapper;

    @Resource
    private ComBaseInfoMapper comBaseInfoMapper;

    @Resource
    private TaskCacheService taskCacheService;

    @Resource
    private CommonService commonService;

    @Resource
    private IdWorker idWorker;

    /**
     * 查询企业华为云会议映射
     *
     * @param comId 企业华为云会议映射ID
     * @return 企业华为云会议映射
     */
    @Override
    public ComHuaweiMeetMatch selectComHuaweiMeetMatchById(Long comId){
        return comHuaweiMeetMatchMapper.selectComHuaweiMeetMatchById(comId);
    }

    /**
     * 查询企业华为云会议映射列表
     *
     * @param comHuaweiMeetMatch 企业华为云会议映射
     * @return 企业华为云会议映射
     */
    @Override
    public List<ComHuaweiMeetMatchDto> selectComHuaweiMeetMatchList(ComHuaweiMeetMatchDto comHuaweiMeetMatch){
        return comHuaweiMeetMatchMapper.selectComHuaweiMeetMatchList(comHuaweiMeetMatch);
    }

    /**
     * 修改企业华为云会议映射
     *
     * @param comHuaweiMeetMatch 企业华为云会议映射
     * @return 结果
     */
    @Override
    public int saveComHuaweiMeetMatch(ComHuaweiMeetMatch comHuaweiMeetMatch) throws PlatFormBusinessException {

        ComBaseInfo comBaseInfo = comBaseInfoMapper.selectComBaseInfoById(comHuaweiMeetMatch.getComId());
        if(comBaseInfo == null){
            throw new PlatFormBusinessException("comId无效");
        }
        if (comHuaweiMeetMatchMapper.getComHuaweiMeetMatchByComId(comHuaweiMeetMatch.getComId()) != null) {
            throw new PlatFormBusinessException("华为云企业账号已存在");
        }
        //自定义的账号密码
        String comCerNo = comBaseInfo.getComCerNo();
        String account = "HW_"+comCerNo;
        String pwd = MD5Util.string2MD5("Saas@"+comCerNo.substring(comCerNo.length()-6,comCerNo.length())).substring(4,16);
        Result result = buildCompanyByAdmin(comBaseInfo.getComName(),account,pwd);
        if(Result.SUCCESS.equals(result.getRetType())){
            String Id = String.valueOf(result.getData());
            comHuaweiMeetMatch.setHuaweiManageUserName(account);
            comHuaweiMeetMatch.setHuaweiComId(Long.parseLong(Id));
            comHuaweiMeetMatch.setHuaweiManagePassWord(pwd);
            comHuaweiMeetMatch.setCreateTime(DateUtils.getNowDate());
            return comHuaweiMeetMatchMapper.save(comHuaweiMeetMatch);
        }else{
            throw new PlatFormBusinessException(result.getMsg());
        }
    }


    /**
     * <b> 管理员创建企业 </b>
     * @author ZXF
     * @create 2020/05/21 0021 15:21
     * @version
     * @注意事项 </b>
     */
    private Result buildCompanyByAdmin(String companyName, String account, String pwd) throws PlatFormBusinessException {
        AddCorpDTO acDTO = new AddCorpDTO();
        AdminDTO adminInfo = new AdminDTO();
        adminInfo.setAccount(account);
        adminInfo.setPwd(pwd);
        adminInfo.setName(companyName);
        adminInfo.setEmail(commonService.getSystemConfig("MEET_BUILD_SP_EMAIL"));
        acDTO.setAdminInfo(adminInfo);
        CorpBasicDTO basicInfo = new CorpBasicDTO();
        basicInfo.setName(companyName);
        acDTO.setBasicInfo(basicInfo);
        /*** 获取华为云会议的accessToken ***/
        String accessToken = commonService.getComManageHuaweiMeetAccessToken(
                commonService.getSystemConfig("MEET_MANAGE_SP_USER")
                , commonService.getSystemConfig("MEET_MANAGE_SP_PWD")
                , commonService.getSystemConfig("MEET_MANAGE_SP_COMID")
                );

        /*** 调用华为云会议接口 ***/
        String url = HuaweiMeetConst.GATEWAY_URL + HuaweiMeetConst.BUILD_CORP;
        String param = JSON.toJSONString(acDTO);
        Map<String, String> headMap = new HashMap<>();
        headMap.put("X-Access-Token", accessToken);
        logger.info(" ========== 管理员创建企业接口");
        logger.info(" ========== 请求地址："+url);
        logger.info(" ========== 请求报文："+param);
        RestRequest restRequest = HttpBuildUtil.buildRestRequest("POST", headMap, param, null);
        RestResponse response = HttpBuildUtil.sendMsg(url, restRequest);
        logger.info(" ========== 返回报文："+response.getEntity());
        //如果返回报文处理resultMsg有值，说明有失败信息
        String resultMsg = HttpBuildUtil.failMsg(response);
        if(StringUtils.isNotEmpty(resultMsg)){
            return Result.fail(resultMsg);
        }
        Map map = HttpBuildUtil.getMapData(response);
        if(map == null){
            if(map.get("error_msg") != null){
              return Result.fail(map.get("error_msg").toString());
            }
            return Result.fail("未获取到企业ID!");
        }
        return Result.success(Result.SUCCESS_MSG,map.get("value"));
    }

    @Override
    public int updateComHuaweiMeetMatch(ComHuaweiMeetMatch comHuaweiMeetMatch){
        if(comHuaweiMeetMatch.getComId() == null){
            comHuaweiMeetMatch.setCreateTime(DateUtils.getNowDate());
            comHuaweiMeetMatch.setComId(idWorker.nextId());
        }else{
            ComHuaweiMeetMatch dbComHuaweiMeetMatch = comHuaweiMeetMatchMapper.selectComHuaweiMeetMatchById(comHuaweiMeetMatch.getComId());
            BeanUtils.copyPropertiesIgnore(comHuaweiMeetMatch,dbComHuaweiMeetMatch,true);
            BeanUtils.copyPropertiesIgnore(dbComHuaweiMeetMatch,comHuaweiMeetMatch,false);
        }
        return comHuaweiMeetMatchMapper.save(comHuaweiMeetMatch);
    }

    /**
     * 删除企业华为云会议映射对象
     *
     * @param ids 需要删除的数据ID
     * @return 结果
     */
    @Override
    public int deleteComHuaweiMeetMatchByIds(String ids){
        return comHuaweiMeetMatchMapper.deleteComHuaweiMeetMatchByIds(Convert.toStrArray(ids));
    }

    /**
     * 删除企业华为云会议映射信息
     *
     * @param comId 企业华为云会议映射ID
     * @return 结果
     */
    @Override
    public int deleteComHuaweiMeetMatchById(Long comId){
        return comHuaweiMeetMatchMapper.deleteComHuaweiMeetMatchById(comId);
    }

    /**
     * 获取企业华为云会议用户信息
     *
     * @param comHuaweiMeetMatch
     * @return
     * @auth llc
     * @date 2019-12-24
     */
    @Override
    public Result getComHuaweiMeet(ComHuaweiMeetMatch comHuaweiMeetMatch) throws PlatFormBusinessException {
        ComHuaweiMeetMatch comHuaweiMeetMatch1 = comHuaweiMeetMatchMapper.getComHuaweiMeetMatchByComId(comHuaweiMeetMatch.getComId());
        return Result.success(Result.SUCCESS_QUERY_MSG, comHuaweiMeetMatch1);
    }

    /**
     * 获取企业华为云会议管理资源
     *
     * @param perHuaweiMeetMatch
     * @return
     * @auth llc
     * @date 2019-12-10
     */
    @Override
    public Result getHuaweiManageResource(PerHuaweiMeetMatch perHuaweiMeetMatch) throws PlatFormBusinessException {
        ComHuaweiMeetMatch comHuaweiMeetMatch = comHuaweiMeetMatchMapper.getComHuaweiMeetMatchByComId(perHuaweiMeetMatch.getComId());
        if (comHuaweiMeetMatch == null) {
            return Result.fail("您所在企业还未购买会议资源");
        }
        PerHuaweiMeetMatch meetMatch = perHuaweiMeetMatchMapper.getPerHuaweiMeetMatchByPerId(perHuaweiMeetMatch.getPerId());
        if (meetMatch == null) {
            return Result.fail("企业管理员还未把您添加到会议使用权限，请联系您所在企业管理员添加");
        }
        /*** 获取华为云会议的accessToken ***/
        String accessToken = taskCacheService.getHuaweiMeetAccessToken(meetMatch.getHuaweiUserName(), meetMatch.getHuaweiPassWord(), perHuaweiMeetMatch.getPerId());
        String url = HuaweiMeetConst.GATEWAY_URL + HuaweiMeetConst.GET_MANAGE_RESOURCE;
        Map<String, String> headMap = new HashMap<>();
        headMap.put("X-Access-Token", accessToken);
        RestRequest restRequest = HttpBuildUtil.buildRestRequest("GET", headMap, null, null);
        RestResponse response = HttpBuildUtil.sendMsg(url, restRequest);
        //如果返回报文处理resultMsg有值，说明有失败信息
        String resultMsg = HttpBuildUtil.failMsg(response);
        if (StringUtils.isNotEmpty(resultMsg)) {
            return Result.fail(resultMsg);
        }
        return Result.success(Result.SUCCESS_QUERY_MSG, HttpBuildUtil.getMapData(response));
    }

    /**
     * <b> 预约会议 </b>
     * @author ZXF
     * @create 2020/01/13 0013 17:18
     * @version
     * @注意事项 </b>
     */
    @Override
    public Result buildHuaweiMeet(RestScheduleConfDTO restScheduleConfDTO) throws PlatFormBusinessException {

        if (restScheduleConfDTO.getMediaTypes() == null) {
            return Result.fail("会议的媒体类型不能为空");
        }
        if (StringUtils.isNotEmpty(restScheduleConfDTO.getAttendeesJSON())) {
            restScheduleConfDTO.setAttendees(JSON.parseArray(restScheduleConfDTO.getAttendeesJSON(), AttendeeDTO.class));
        }
        PerHuaweiMeetMatch meetMatch = perHuaweiMeetMatchMapper.getPerHuaweiMeetMatchByPerId(restScheduleConfDTO.getPerId());
        if (meetMatch == null) {
            return Result.fail("企业管理员还未把您添加到会议使用权限，请联系您所在企业管理员添加");
        }

        /*** 获取华为云会议的accessToken ***/
        String accessToken = taskCacheService.getHuaweiMeetAccessToken(meetMatch.getHuaweiUserName(), meetMatch.getHuaweiPassWord(), meetMatch.getPerId());

        /*** 调用华为云会议接口 ***/
        String url = HuaweiMeetConst.GATEWAY_URL + HuaweiMeetConst.ADD_MEET;
        Map<String, String> headMap = new HashMap<>();
        headMap.put("X-Access-Token", accessToken);
        String param = JSON.toJSONString(restScheduleConfDTO);
        logger.info(" ========== 预约会议接口");
        logger.info(" ========== 请求地址：" + url);
        logger.info(" ========== 请求报文：" + param);
        RestRequest restRequest = HttpBuildUtil.buildRestRequest("POST", headMap, param, null);
        RestResponse response = HttpBuildUtil.sendMsg(url, restRequest);
        //如果返回报文处理resultMsg有值，说明有失败信息
        logger.info(" ========== 返回报文：" + response.getEntity());
        String resultMsg = HttpBuildUtil.failMsg(response);
        if (StringUtils.isNotEmpty(resultMsg)) {
            return Result.fail(resultMsg);
        }
        JSONArray arr = JSON.parseArray(response.getEntity());
        return Result.success(Result.SUCCESS_QUERY_MSG, arr);
    }

    /**
     * <b> 用户查询自己的信息 </b>
     * @author ZXF
     * @create 2020/01/13 0013 17:18
     * @version
     * @注意事项 </b>
     */
    @Override
    public Result getHuaweiUserDetail(PerHuaweiMeetMatch perHuaweiMeetMatch) throws PlatFormBusinessException {

        PerHuaweiMeetMatch meetMatch = perHuaweiMeetMatchMapper.getPerHuaweiMeetMatchByPerId(perHuaweiMeetMatch.getPerId());
        if (meetMatch == null) {
            return Result.fail("企业管理员还未把您添加到会议使用权限，请联系您所在企业管理员添加");
        }

        /*** 获取华为云会议的accessToken ***/
        String accessToken = taskCacheService.getHuaweiMeetAccessToken(meetMatch.getHuaweiUserName(), meetMatch.getHuaweiPassWord(), meetMatch.getPerId());

        /*** 调用华为云会议接口 ***/
        String url = HuaweiMeetConst.GATEWAY_URL + HuaweiMeetConst.GET_USER_DETAIL;
        Map<String, String> headMap = new HashMap<>();
        headMap.put("X-Access-Token", accessToken);
        logger.info(" ========== 用户查询自己的信息");
        logger.info(" ========== 请求地址：" + url);
        RestRequest restRequest = HttpBuildUtil.buildRestRequest("GET", headMap, null, null);
        RestResponse response = HttpBuildUtil.sendMsg(url, restRequest);
        logger.info(" ========== 返回报文：" + response.getEntity());
        //如果返回报文处理resultMsg有值，说明有失败信息
        String resultMsg = HttpBuildUtil.failMsg(response);
        if (StringUtils.isNotEmpty(resultMsg)) {
            return Result.fail(resultMsg);
        }
        return Result.success(Result.SUCCESS_QUERY_MSG, HttpBuildUtil.getMapData(response));
    }


    /**
     * 获取企业华为专用云会议室
     *
     * @param dto
     * @return
     * @auth llc
     * @date 2020-02-17
     */
    @Override
    public Result getHuaweiCloudMeetRoomList(PerHuaweiMeetMatchDto dto) throws PlatFormBusinessException {
        if (dto.getPageNumber() == null) {
            return Result.fail("第几页不能为空");
        }
        if (dto.getPageSize() == null) {
            return Result.fail("每页多少条不能为空");
        }
        ComHuaweiMeetMatch comHuaweiMeetMatch = comHuaweiMeetMatchMapper.getComHuaweiMeetMatchByComId(dto.getComId());
        if (comHuaweiMeetMatch == null) {
            return Result.fail("您所在企业还未购买会议资源");
        }
        /*** 获取华为云会议的accessToken ***/
        String accessToken = taskCacheService.getComManageHuaweiMeetAccessToken(comHuaweiMeetMatch.getHuaweiManageUserName(), comHuaweiMeetMatch.getHuaweiManagePassWord(), dto.getComId());
        if (StringUtils.isEmpty(accessToken)) {
            return Result.fail("华为云会议鉴权失败");
        }
        String url = HuaweiMeetConst.GATEWAY_URL + HuaweiMeetConst.GET_CLOUD_MEET_ROOM_LIST;
        /*** 调用华为云会议接口 ***/
        Map<String, String> map = new HashMap<>();
        map.put("offset", String.valueOf(dto.getPageNumber() - 1));
        map.put("limit", dto.getPageSize().toString());
        if (StringUtils.isNotEmpty(dto.getKeyWord())) {
            map.put("searchKey", dto.getKeyWord());
        }
        map.put("status", dto.getStatus());
        Map<String, String> headMap = new HashMap<>();
        headMap.put("X-Access-Token", accessToken);
        RestRequest restRequest = HttpBuildUtil.buildRestRequest("GET", headMap, null, map);
        RestResponse response = HttpBuildUtil.sendMsg(url, restRequest);
        //如果返回报文处理resultMsg有值，说明有失败信息
        String resultMsg = HttpBuildUtil.failMsg(response);
        if (StringUtils.isNotEmpty(resultMsg)) {
            return Result.fail(resultMsg);
        }
        return Result.success(Result.SUCCESS_QUERY_MSG, HttpBuildUtil.getMapData(response));
    }

    /**
     * <b> 修改用户云会议室 </b>
     * @author ZXF
     * @create 2020/02/19 0019 10:55
     * @version
     * @注意事项 </b>
     */
    @Override
    public Result updateHuaweiCloudMemberVmr(ModVmrDto dto) throws PlatFormBusinessException {
        if (StringUtils.isEmpty(dto.getId())) {
            return Result.fail("云会议室唯一标识不能为空");
        }
        dto.setBool();
        PerHuaweiMeetMatch meetMatch = perHuaweiMeetMatchMapper.getPerHuaweiMeetMatchByPerId(dto.getPerId());
        if (meetMatch == null) {
            return Result.fail("企业管理员还未把您添加到会议使用权限，请联系您所在企业管理员添加");
        }

        /*** 获取华为云会议的accessToken ***/
        String accessToken = taskCacheService.getHuaweiMeetAccessToken(meetMatch.getHuaweiUserName(), meetMatch.getHuaweiPassWord(), meetMatch.getPerId());

        /*** 调用华为云会议接口 ***/
        String url = HuaweiMeetConst.GATEWAY_URL + HuaweiMeetConst.UPDATE_MEMBER_VMR + dto.getId();
        Map<String, String> headMap = new HashMap<>();
        headMap.put("X-Access-Token", accessToken);
        Map<String, Object> map = new HashMap<>();
        map.put("vmrName", dto.getVmrName());
        map.put("gustPwd", dto.getGustPwd());
        map.put("chairPwd", dto.getChairPwd());
        map.put("allowGustFirst", dto.getAllowGustFirst());
        map.put("gustFirstNotice", dto.getGustFirstNotice());
        String param = JSON.toJSONString(map);
        logger.info(" ========== 修改用户云会议室接口");
        logger.info(" ========== 请求地址：" + url);
        logger.info(" ========== 请求报文：" + param);
        RestRequest restRequest = HttpBuildUtil.buildRestRequest("PUT", headMap, param, null);
        RestResponse response = HttpBuildUtil.sendMsg(url, restRequest);
        logger.info(" ========== 返回报文：" + response.getEntity());
        //如果返回报文处理resultMsg有值，说明有失败信息
        String resultMsg = HttpBuildUtil.failMsg(response);
        if (StringUtils.isNotEmpty(resultMsg)) {
            return Result.fail(resultMsg);
        }
        return Result.success(Result.SUCCESS_MSG);
    }

    /**
     * <b> 回收专用云会议室 </b>
     * @author ZXF
     * @create 2020/02/19 0019 10:55
     * @version
     * @注意事项 </b>
     */
    @Override
    public Result recycleFromMember(ModVmrDto dto) throws PlatFormBusinessException {
        if (StringUtils.isEmpty(dto.getAccount())) {
            return Result.fail("用户账号不能为空");
        }
        if (StringUtils.isEmpty(dto.getVmrIdList())) {
            return Result.fail("云会议室唯一ID列表不能为空");
        }
        ComHuaweiMeetMatch comHuaweiMeetMatch = comHuaweiMeetMatchMapper.getComHuaweiMeetMatchByComId(dto.getComId());
        if (comHuaweiMeetMatch == null) {
            return Result.fail("您所在企业还未购买会议资源");
        }
        /*** 获取华为云会议的accessToken ***/
        String accessToken = taskCacheService.getComManageHuaweiMeetAccessToken(comHuaweiMeetMatch.getHuaweiManageUserName(), comHuaweiMeetMatch.getHuaweiManagePassWord(), dto.getComId());

        /*** 调用华为云会议接口 ***/
        String url = HuaweiMeetConst.GATEWAY_URL + HuaweiMeetConst.RECYCLE_FROM_MEMBER + dto.getAccount();
        Map<String, String> headMap = new HashMap<>();
        headMap.put("X-Access-Token", accessToken);
        String param = dto.getVmrIdList();
        logger.info(" ========== 回收专用云会议室接口");
        logger.info(" ========== 请求地址：" + url);
        logger.info(" ========== 请求报文：" + param);
        RestRequest restRequest = HttpBuildUtil.buildRestRequest("POST", headMap, param, null);
        RestResponse response = HttpBuildUtil.sendMsg(url, restRequest);
        logger.info(" ========== 返回报文：" + response.getEntity());
        //如果返回报文处理resultMsg有值，说明有失败信息
        String resultMsg = HttpBuildUtil.failMsg(response);
        if (StringUtils.isNotEmpty(resultMsg)) {
            return Result.fail(resultMsg);
        }
        return Result.success(Result.SUCCESS_MSG);
    }


    /**
     * <b> 分配专用云会议室 </b>
     * @author ZXF
     * @create 2020/02/19 0019 10:55
     * @version
     * @注意事项 </b>
     */
    @Override
    public Result assignToMember(ModVmrDto dto) throws PlatFormBusinessException {
        if (StringUtils.isEmpty(dto.getAccount())) {
            return Result.fail("用户账号不能为空");
        }
        if (StringUtils.isEmpty(dto.getVmrIdList())) {
            return Result.fail("云会议室唯一ID列表不能为空");
        }
        ComHuaweiMeetMatch comHuaweiMeetMatch = comHuaweiMeetMatchMapper.getComHuaweiMeetMatchByComId(dto.getComId());
        if (comHuaweiMeetMatch == null) {
            return Result.fail("您所在企业还未购买会议资源");
        }
        /*** 获取华为云会议的accessToken ***/
        String accessToken = taskCacheService.getComManageHuaweiMeetAccessToken(comHuaweiMeetMatch.getHuaweiManageUserName(), comHuaweiMeetMatch.getHuaweiManagePassWord(), dto.getComId());

        /*** 调用华为云会议接口 ***/
        String url = HuaweiMeetConst.GATEWAY_URL + HuaweiMeetConst.ASSIGN_TO_MEMBER + dto.getAccount();
        Map<String, String> headMap = new HashMap<>();
        headMap.put("X-Access-Token", accessToken);
        String param = dto.getVmrIdList();
        logger.info(" ========== 分配专用云会议室接口");
        logger.info(" ========== 请求地址：" + url);
        logger.info(" ========== 请求报文：" + param);
        RestRequest restRequest = HttpBuildUtil.buildRestRequest("POST", headMap, param, null);
        RestResponse response = HttpBuildUtil.sendMsg(url, restRequest);
        logger.info(" ========== 返回报文：" + response.getEntity());
        //如果返回报文处理resultMsg有值，说明有失败信息
        String resultMsg = HttpBuildUtil.failMsg(response);
        if (StringUtils.isNotEmpty(resultMsg)) {
            return Result.fail(resultMsg);
        }
        return Result.success(Result.SUCCESS_MSG);
    }

    /**
     * <b> 删除专用云会议室 </b>
     * @author ZXF
     * @create 2020/02/19 0019 10:55
     * @version
     * @注意事项 </b>
     */
    @Override
    public Result deleteMemberVmr(ModVmrDto dto) throws PlatFormBusinessException {
        if (StringUtils.isEmpty(dto.getVmrIdList())) {
            return Result.fail("云会议室唯一ID列表不能为空");
        }
        ComHuaweiMeetMatch comHuaweiMeetMatch = comHuaweiMeetMatchMapper.getComHuaweiMeetMatchByComId(dto.getComId());
        if (comHuaweiMeetMatch == null) {
            return Result.fail("您所在企业还未购买会议资源");
        }
        /*** 获取华为云会议的accessToken ***/
        String accessToken = taskCacheService.getComManageHuaweiMeetAccessToken(comHuaweiMeetMatch.getHuaweiManageUserName(), comHuaweiMeetMatch.getHuaweiManagePassWord(), dto.getComId());

        /*** 调用华为云会议接口 ***/
        String url = HuaweiMeetConst.GATEWAY_URL + HuaweiMeetConst.DELETE_MEMBER_VMR;
        Map<String, String> headMap = new HashMap<>();
        headMap.put("X-Access-Token", accessToken);
        String param = dto.getVmrIdList();
        logger.info(" ========== 删除专用云会议室接口");
        logger.info(" ========== 请求地址：" + url);
        logger.info(" ========== 请求报文：" + param);
        RestRequest restRequest = HttpBuildUtil.buildRestRequest("POST", headMap, param, null);
        RestResponse response = HttpBuildUtil.sendMsg(url, restRequest);
        logger.info(" ========== 返回报文：" + response.getEntity());
        //如果返回报文处理resultMsg有值，说明有失败信息
        String resultMsg = HttpBuildUtil.failMsg(response);
        if (StringUtils.isNotEmpty(resultMsg)) {
            return Result.fail(resultMsg);
        }
        return Result.success(Result.SUCCESS_MSG);
    }

    /**
     * <b> 分页查询用户云会议室 </b>
     * @author ZXF
     * @create 2020/02/20 0020 10:43
     * @version
     * @注意事项 </b>
     */
    @Override
    public Result getHuaweiCloudMemberVmrList(ModVmrDto dto) throws PlatFormBusinessException {
        if (dto.getPageNumber() == null) {
            return Result.fail("第几页不能为空");
        }
        if (dto.getPageSize() == null) {
            return Result.fail("每页多少条不能为空");
        }
        PerHuaweiMeetMatch meetMatch = perHuaweiMeetMatchMapper.getPerHuaweiMeetMatchByPerId(dto.getPerId());
        if (meetMatch == null) {
            return Result.fail("企业管理员还未把您添加到会议使用权限，请联系您所在企业管理员添加");
        }

        /*** 获取华为云会议的accessToken ***/
        String accessToken = taskCacheService.getHuaweiMeetAccessToken(meetMatch.getHuaweiUserName(), meetMatch.getHuaweiPassWord(), meetMatch.getPerId());
        String url = HuaweiMeetConst.GATEWAY_URL + HuaweiMeetConst.GET_MEMBER_VMR_LIST;
        /*** 调用华为云会议接口 ***/
        Map<String, String> map = new HashMap<>();
        map.put("offset", String.valueOf(dto.getPageNumber() - 1));
        map.put("limit", dto.getPageSize().toString());
        if (StringUtils.isNotEmpty(dto.getSearchKey())) {
            map.put("searchKey", dto.getSearchKey());
        }
        Boolean specialVmr = null;
        if (StringUtils.isNotEmpty(dto.getSpecialVmr())) {
            specialVmr = "1".equals(dto.getSpecialVmr()) ? true : "0".equals(dto.getSpecialVmr()) ? false:null;
        }
        map.put("specialVmr", specialVmr == null ? "" : String.valueOf(specialVmr));
        Map<String, String> headMap = new HashMap<>();
        headMap.put("X-Access-Token", accessToken);
        RestRequest restRequest = HttpBuildUtil.buildRestRequest("GET", headMap, null, map);
        RestResponse response = HttpBuildUtil.sendMsg(url, restRequest);
        //如果返回报文处理resultMsg有值，说明有失败信息
        String resultMsg = HttpBuildUtil.failMsg(response);
        if (StringUtils.isNotEmpty(resultMsg)) {
            return Result.fail(resultMsg);
        }
        return Result.success(Result.SUCCESS_QUERY_MSG, HttpBuildUtil.getMapData(response));
    }


    /**
     * 刷新登陆
     *
     * @param comId
     * @return
     * @auth wangjianjun
     * @date 2020-05-11
     */
    @Override
    public int refreshComHuaweiMeet(Long comId){
        commonService.deleteRedisByKey("huaweiMeetCom"+comId);
        return 1;
    }

}