package com.bcxin.platform.service.attend.impl;

import com.bcxin.platform.common.constant.Constants;
import com.bcxin.platform.common.core.text.Convert;
import com.bcxin.platform.common.exception.BusinessException;
import com.bcxin.platform.common.utils.DateUtils;
import com.bcxin.platform.common.utils.IdWorker;
import com.bcxin.platform.common.utils.bean.BeanUtils;
import com.bcxin.platform.domain.attend.Attend;
import com.bcxin.platform.domain.attend.AttendPer;
import com.bcxin.platform.domain.company.PerBaseInfo;
import com.bcxin.platform.dto.app.AppAttendPerDto;
import com.bcxin.platform.dto.attend.AttendPerDto;
import com.bcxin.platform.mapper.attend.AttendMapper;
import com.bcxin.platform.mapper.attend.AttendPerMapper;
import com.bcxin.platform.mapper.attend.AttendSchedulDetailMapper;
import com.bcxin.platform.service.attend.AttendPerService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 考勤人员Service业务层处理
 * 
 * @author lin
 * @date 2021-08-10
 */
@Service
@Transactional("transactionManager")
public class AttendPerServiceImpl implements AttendPerService{

    @Autowired
    private AttendMapper attendMapper;

    @Autowired
    private AttendPerMapper attendPerMapper;

    @Autowired
    private AttendSchedulDetailMapper attendSchedulMapper;

    @Resource
    private IdWorker idWorker;

    /**
     * 查询考勤人员
     * 
     * @param id 考勤人员ID
     * @return 考勤人员
     */
    @Override
    public AttendPer findById(Long id){
        return attendPerMapper.findById(id);
    }

    /**
     * 查询考勤人员列表
     * 
     * @param attendPer 考勤人员
     * @return 考勤人员
     */
    @Override
    public List<AttendPerDto> selectList(AttendPerDto attendPer){
        return attendPerMapper.selectList(attendPer);
    }

    /**
     * 修改考勤人员
     * 
     * @param attendPer 考勤人员
     * @return 结果
     */
    @Override
    public int update(AttendPer attendPer){
        Attend attend = attendMapper.findById(attendPer.getAttendId());
        if(attend == null){
            throw new BusinessException("考勤套id错误，对应考勤套不存在！");
        }else if(Constants.N.equals(attend.getActive())){
            throw new BusinessException("考勤套未启用！");
        }else if(Constants.Y.equals(attend.getIsDelete())){
            throw new BusinessException("考勤套已删除！");
        }
        //重复判断
        AttendPer checkAttendPer = attendPerMapper.findByPerId(attendPer.getPerId());
        if(checkAttendPer != null && !checkAttendPer.getId().equals(attendPer.getId())){
            throw new BusinessException("该保安员已维护考勤套，请勿重复维护！");
        }

        if(attendPer.getId() == null){
            attendPer.setCreateTime(DateUtils.getNowDate());
            attendPer.setId(idWorker.nextId());
            attendPer.setIsDelete(Constants.N);
        }else{
            AttendPer dbAttendPer = attendPerMapper.findById(attendPer.getId());

            if(dbAttendPer.getAttendId().longValue() == attendPer.getAttendId().longValue()){
                throw new BusinessException("考勤套没有变更，无需操作保存");
            }
            dbAttendPer.setUpdateTime(DateUtils.getNowDate());
            //删除今天之后的排班记录
            attendPerMapper.removeSchedul(dbAttendPer);

            BeanUtils.copyPropertiesIgnore(attendPer,dbAttendPer,true);
            BeanUtils.copyPropertiesIgnore(dbAttendPer,attendPer,false);
        }
        attendPer.setUpdateTime(DateUtils.getNowDate());
        return attendPerMapper.save(attendPer);
    }

    /**
     * 删除考勤人员对象
     * 
     * @param ids 需要删除的数据ID
     * @return 结果
     */
    @Override
    public int deleteByIds(String ids){
        //删除相关排班信息
        //删除今天之后的，不含今天
        attendSchedulMapper.removeByPerIds(Convert.toStrArray(ids));
        return attendPerMapper.deleteByIds(Convert.toStrArray(ids));
    }

    @Override
    public List<PerBaseInfo> getDepartPersonList(AttendPerDto searchDto) {
        //加上数据权限
        if(!searchDto.isDomainAdmin() && (searchDto.getAdminDepartIds()== null || searchDto.getAdminDepartIds().size() == 0)){
            return new ArrayList<>();
        }
        return attendPerMapper.getDepartPersonList(searchDto);
    }

    @Override
    public List<AppAttendPerDto> selectListForPer(AppAttendPerDto perDto) {
        return attendPerMapper.selectListForPer(perDto);
    }

    @Override
    public int save(AttendPer attendPer) throws CloneNotSupportedException {
        Attend attend = attendMapper.findById(attendPer.getAttendId());
        if(attend == null){
            throw new BusinessException("考勤套id错误，对应考勤套不存在！");
        }else if(Constants.N.equals(attend.getActive())){
            throw new BusinessException("考勤套未启用！");
        }else if(Constants.Y.equals(attend.getIsDelete())){
            throw new BusinessException("考勤套已删除！");
        }

        attendPer.setCreateTime(DateUtils.getNowDate());
        attendPer.setUpdateTime(DateUtils.getNowDate());
        attendPer.setIsDelete(Constants.N);
        //重复判断
        List<AttendPer> perList = new ArrayList<>();
        for (Long perId : attendPer.getPerIds()) {
            AttendPer addAttendPer = attendPer.clone();
            addAttendPer.setId(idWorker.nextId());
            addAttendPer.setPerId(perId);
            perList.add(addAttendPer);
        }
        List<PerBaseInfo> dbAttendPerList = attendPerMapper.findByBatchPerId(perList);
        if(dbAttendPerList != null && dbAttendPerList.size() > 0){
            String perNames = dbAttendPerList.stream().map(PerBaseInfo::getName).collect(Collectors.toList()).stream().collect(Collectors.joining(", "));
            throw new BusinessException("保安员("+perNames+")已维护考勤套，请勿重复维护！");
        }

        return attendPerMapper.saveBatch(perList);
    }

    @Override
    public List<AppAttendPerDto> getUnSchedulPersonList(AppAttendPerDto perDto) {
        return attendPerMapper.getUnSchedulPersonList(perDto);
    }

    @Override
    public int deleteByPerIds(AppAttendPerDto attendPerDto) {
        //删除相关排班信息
        //删除今天之后的，不含今天
        attendSchedulMapper.deleteByPerIds(attendPerDto);
        return attendPerMapper.deleteByPerIds(attendPerDto);
    }
}
